package ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.Configuration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationResponse;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdate;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.PhrasesRecord;

import static ru.yandex.direct.core.entity.moderation.model.ModerationDecision.No;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;

@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class UpdateMinusRegionsOp extends BulkBannersOp {
    private final Map<Long, List<Long>> minusGeo = new HashMap<>();
    private final List<Long> allBids = new ArrayList<>();
    private final Set<Long> declinedBids = new HashSet<>();
    private final BannerModerationRepository bannerModerationRepository;
    private final BannerRelationsRepository bannerRelationsRepository;

    @Autowired
    public UpdateMinusRegionsOp(BannerModerationRepository bannerModerationRepository,
                                BannerRelationsRepository bannerRelationsRepository) {
        this.bannerModerationRepository = bannerModerationRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
    }

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, BannerModerationResponse response) {
        allBids.add(response.getMeta().getBannerId());

        if (!CollectionUtils.isEmpty(response.getResult().getMinusRegions())) {
            minusGeo.put(response.getMeta().getBannerId(), response.getResult().getMinusRegions());
        }

        if (response.getResult().getVerdict() == No) {
            declinedBids.add(response.getMeta().getBannerId());
        }
    }

    //- если объявление отклонено, и по нему есть/были минус-регионы - останавливаем это объявление
    private void rejectDeclinedBannersWithMinusGeo(BulkUpdateHolder bulkUpdateHolder, Configuration configuration) {
        Map<Long, List<Long>> currentMinusGeo =
                bannerModerationRepository.getCurrentBannersMinusGeo(configuration.dsl(), declinedBids);

        Set<Long> justGotMinusRegions = Sets.intersection(declinedBids, minusGeo.keySet());

        BulkUpdate<BannersRecord, Long> updater = bulkUpdateHolder.get(BANNERS.BID);

        for (Long bid : Iterables.concat(currentMinusGeo.keySet(), justGotMinusRegions)) {
            updater.forId(bid)
                    .set(BANNERS.STATUS_MODERATE, BannersStatusmoderate.No)
                    .set(BANNERS.STATUS_POST_MODERATE, BannersStatuspostmoderate.Rejected);
        }
    }

    //- если по баннеру пришли минус-регионы, и группа отклонена - останавливаем показы всех баннеров группы
    private void rejectDeclinedGroupsWithMinusGeo(BulkUpdateHolder bulkUpdateHolder, Configuration configuration) {
        Set<Long> bannersWithDeclinedGroups =
                bannerRelationsRepository.bannersWithDeclinedAdGroups(configuration, minusGeo.keySet());

        Collection<Long> adgroupIds =
                bannerRelationsRepository.getAdGroupIdsByBannerIds(configuration.dsl(), bannersWithDeclinedGroups)
                        .values();

        BulkUpdate<PhrasesRecord, Long> updater = bulkUpdateHolder.get(PHRASES.PID);

        for (Long pid : adgroupIds.stream().distinct().collect(Collectors.toList())) {
            updater.forId(pid).set(PHRASES.STATUS_POST_MODERATE, PhrasesStatuspostmoderate.Rejected);
        }
    }

    /*
        https://wiki.yandex-team.ru/AnastasijaBalakina/Geolegal/

        Логика остановки показа предыдущей версии:
        - если объявление отклонено, и по нему есть/были минус-регионы - останавливаем это объявление
        - если группа отклонена, и по ее объявлениям есть/были минус-регионы - останавливаем показы всех объявлений
        группы
        - если по баннеру пришли минус-регионы, и группа отклонена - останавливаем показы всех баннеров группы
     */

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        super.flush(configuration, bulkUpdateHolder);
        rejectDeclinedBannersWithMinusGeo(bulkUpdateHolder, configuration);
        rejectDeclinedGroupsWithMinusGeo(bulkUpdateHolder, configuration);
        bannerModerationRepository.deleteMinusGeo(configuration.dsl(), allBids);
        bannerModerationRepository.insertMinusGeo(configuration, minusGeo);
    }
}
