package ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jooq.Configuration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationResponse;
import ru.yandex.direct.core.entity.moderation.model.ModerationDecision;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;

import static ru.yandex.direct.core.entity.moderation.model.ModerationDecision.No;
import static ru.yandex.direct.core.entity.moderation.model.ModerationDecision.Yes;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;

@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class UpdateStatusModerateAndPostModerateOp extends BulkBannersOp {
    private final ModerationDiagService moderationDiagService;
    private final BannerModerationRepository bannerModerationRepository;

    private Set<Long> acceptedBanners = new HashSet<>();
    private Set<Long> rejectedBanners = new HashSet<>();
    private Set<Long> rejectedBannersByCriticalReason = new HashSet<>();

    @Autowired
    public UpdateStatusModerateAndPostModerateOp(ModerationDiagService moderationDiagService,
                                                 BannerModerationRepository bannerModerationRepository) {
        this.moderationDiagService = moderationDiagService;
        this.bannerModerationRepository = bannerModerationRepository;
    }

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, BannerModerationResponse response) {
        ModerationDecision statusmoderate = response.getResult().getVerdict();

        if (statusmoderate == Yes) {
            acceptedBanners.add(response.getMeta().getBannerId());
        } else if (statusmoderate == No) {
            rejectedBanners.add(response.getMeta().getBannerId());
            if (isCriticalReason(response)) {
                rejectedBannersByCriticalReason.add(response.getMeta().getBannerId());
            }
        }
    }

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        Map<Long, BannersStatusmoderate> rejectedBannerToStatusModerate =
                bannerModerationRepository.getBannersStatusModerate(configuration, rejectedBanners);

        for (Long bid : acceptedBanners) {
            bulkUpdateHolder
                    .get(BANNERS.BID)
                    .forId(bid)
                    .set(BANNERS.STATUS_MODERATE, BannersStatusmoderate.Yes)
                    .set(BANNERS.STATUS_POST_MODERATE, BannersStatuspostmoderate.Yes);
        }

        for (Long bid : rejectedBanners) {
            boolean hasCriticalReason = rejectedBannersByCriticalReason.contains(bid);
            // Если текущая версия уже была принята, но пришло отклонение, то её надо остановить.
            boolean previouslyAccepted = rejectedBannerToStatusModerate.get(bid) == BannersStatusmoderate.Yes;
            boolean needStop = hasCriticalReason || previouslyAccepted;
            bulkUpdateHolder
                    .get(BANNERS.BID)
                    .forId(bid)
                    .set(BANNERS.STATUS_MODERATE, BannersStatusmoderate.No)
                    .set(BANNERS.STATUS_POST_MODERATE, needStop ? BannersStatuspostmoderate.Rejected : BannersStatuspostmoderate.No);
        }
    }

    private boolean isCriticalReason(BannerModerationResponse response) {

        Map<Long, ModerationDiag> reasonMap = moderationDiagService.get(ModerationDiagType.COMMON);

        if (response.getResult() == null || response.getResult().getReasons() == null) {
            return false;
        }

        return response.getResult().getReasons().stream()
                .anyMatch(el -> reasonMap.containsKey(el) && reasonMap.get(el).getUnbanIsProhibited());
    }

}
