package ru.yandex.direct.core.entity.moderation.service.receiving.operations.common;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jooq.Configuration;

import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResultResponse;
import ru.yandex.direct.core.entity.moderation.model.BaseModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationDecision;
import ru.yandex.direct.core.entity.moderation.model.ModerationResult;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.ModerationResponseProcessingOp;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReason;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonDetailed;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusModerate;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusPostModerate;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusSending;
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository;

public abstract class InsertOrUpdateModerationReasonsOp<T extends AbstractModerationResultResponse<?
        extends BaseModerationMeta, ? extends ModerationResult>> implements ModerationResponseProcessingOp<T> {

    private final Map<Long, ModerationReason> modReasons = new HashMap<>();
    private final ModerationReasonObjectType moderationReasonObjectType;

    private final ModerationReasonRepository moderationReasonRepository;

    private final boolean saveYesModReasons;

    public InsertOrUpdateModerationReasonsOp(ModerationReasonRepository moderationReasonRepository,
                                             ModerationReasonObjectType moderationReasonObjectType,
                                             boolean saveYesModReasons) {
        this.moderationReasonObjectType = moderationReasonObjectType;
        this.moderationReasonRepository = moderationReasonRepository;
        this.saveYesModReasons = saveYesModReasons;
    }

    protected abstract ReasonsInfo parseResponse(T response);

    public static class ReasonsInfo {
        private final long objectId;
        private final long clientId;
        private final long campaignId;
        private final List<ModerationReasonDetailed> reasons;

        public ReasonsInfo(long objectId, long campaignId, long clientId, List<ModerationReasonDetailed> reasons) {
            this.objectId = objectId;
            this.campaignId = campaignId;
            this.clientId = clientId;
            this.reasons = reasons;
        }

        public long getCampaignId() {
            return campaignId;
        }

        public long getObjectId() {
            return objectId;
        }

        public long getClientId() {
            return clientId;
        }

        public List<ModerationReasonDetailed> getReasons() {
            return reasons;
        }

    }

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, T response) {
        ReasonsInfo reasonsInfo = parseResponse(response);

        if (response.getResult().getVerdict() == ModerationDecision.No) {
            modReasons.put(reasonsInfo.getObjectId(),
                    createNoModReasons(reasonsInfo.getObjectId(),
                            reasonsInfo.getCampaignId(),
                            moderationReasonObjectType,
                            reasonsInfo.getClientId(),
                            reasonsInfo.getReasons()));
        }
        if (saveYesModReasons && response.getResult().getVerdict() == ModerationDecision.Yes) {
            modReasons.put(reasonsInfo.getObjectId(),
                    createYesModReasons(reasonsInfo.getObjectId(),
                            reasonsInfo.getCampaignId(),
                            moderationReasonObjectType,
                            reasonsInfo.getClientId()));
        }
    }

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        moderationReasonRepository.insertOrUpdateModerationReasons(configuration, modReasons.values());
    }

    private ModerationReason createYesModReasons(long id, long campaignId, ModerationReasonObjectType type,
                                                 long clientId) {
        return createModReasons(id, campaignId, type, clientId, ModerationReasonStatusModerate.YES,
                Collections.emptyList());
    }

    private ModerationReason createNoModReasons(long id, long campaignId, ModerationReasonObjectType type,
                                                long clientId,
                                                List<ModerationReasonDetailed> reasons) {
        return createModReasons(id, campaignId, type, clientId, ModerationReasonStatusModerate.NO,
                reasons);
    }

    private ModerationReason createModReasons(long id, long campaignId, ModerationReasonObjectType type,
                                              long clientId,
                                              ModerationReasonStatusModerate statusModerate,
                                              List<ModerationReasonDetailed> reasons) {
        return new ModerationReason()
                .withObjectId(id)
                .withObjectType(type)
                .withClientId(clientId)
                .withCampaignId(campaignId)
                .withStatusSending(ModerationReasonStatusSending.NO)
                .withStatusModerate(statusModerate)
                .withStatusPostModerate(ModerationReasonStatusPostModerate.NO)
                .withCreatedAt(LocalDateTime.now())
                .withReasons(reasons);
    }

    protected Map<Long, ModerationReason> getModReasons() {
        return modReasons;
    }
}
