package ru.yandex.direct.core.entity.moderation.service.receiving.operations.common;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.type.href.BannerDomainRepository;
import ru.yandex.direct.core.entity.domain.model.ApiDomainStat;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResultResponse;
import ru.yandex.direct.core.entity.moderation.model.BaseBannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationResult;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.ModerationResponseProcessingOp;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.moderation.model.ModerationDecision.Yes;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class UpdateApiDomainStatOp<T extends AbstractModerationResultResponse<? extends BaseBannerModerationMeta, ?
        extends ModerationResult>>
        implements ModerationResponseProcessingOp<T> {

    private final DomainRepository domainRepository;
    private final BannerDomainRepository bannerDomainRepository;
    private final ModerationDiagService moderationDiagService;

    private List<Stat> stats = new ArrayList<>();

    @Autowired
    public UpdateApiDomainStatOp(DomainRepository domainRepository,
                                 BannerDomainRepository bannerDomainRepository,
                                 ModerationDiagService moderationDiagService) {
        this.domainRepository = domainRepository;
        this.bannerDomainRepository = bannerDomainRepository;
        this.moderationDiagService = moderationDiagService;
    }

    @Override
    public void consume(BulkUpdateHolder bulkUpdateHolder, T response) {
        long badReasonsCount = countBadReasons(response.getResult().getReasons());
        stats.add(
                new Stat(response.getMeta().getBannerId(), response.getResult().getVerdict() == Yes, badReasonsCount));
    }

    private long countBadReasons(List<Long> reasons) {
        if (reasons == null || reasons.isEmpty()) {
            return 0;
        }

        var allReasons = moderationDiagService.get(ModerationDiagType.COMMON);

        // если прислали неизвестную причину, здесь будем считать, что это критичная причина, и посчитаем её
        return reasons.stream()
                .filter(r -> !allReasons.containsKey(r) || allReasons.get(r).getStrongReason())
                .count();
    }

    @Override
    public void flush(Configuration configuration, BulkUpdateHolder bulkUpdateHolder) {
        Map<Long, String> domainsByBids =
                bannerDomainRepository.getDomainsForStat(configuration.dsl(), mapList(stats, s -> s.bid));

        List<ApiDomainStat> domainStatUpdates = StreamEx.of(stats)
                .mapToEntry(stat -> domainsByBids.get(stat.bid), identity())
                .collapseKeys()
                .filterKeys(Objects::nonNull)
                .mapKeyValue(this::collectStatForDomain)
                .toList();

        if (!domainStatUpdates.isEmpty()) {
            domainRepository.updateDomainsStat(domainStatUpdates);
        }
    }

    private ApiDomainStat collectStatForDomain(String domain, List<Stat> stats) {
        ApiDomainStat result = new ApiDomainStat().withFilterDomain(domain)
                .withAcceptedItems(0L).withDeclinedItems(0L).withBadReasons(0L);
        stats.forEach(s -> {
            if (s.accepted) {
                result.setAcceptedItems(result.getAcceptedItems() + 1);
            } else {
                result.setDeclinedItems(result.getDeclinedItems() + 1);
            }
            result.setBadReasons(result.getBadReasons() + s.badReasonsCount);
        });
        return result;
    }

    static class Stat {
        private final long bid;
        private final boolean accepted;
        private final long badReasonsCount;

        public Stat(long bid, boolean accepted, long badReasonsCount) {
            this.bid = bid;
            this.accepted = accepted;
            this.badReasonsCount = badReasonsCount;
        }
    }
}
