package ru.yandex.direct.core.entity.moderation.service.receiving.processing_configurations;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import org.jooq.Configuration;
import org.springframework.context.ApplicationContext;

import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResultResponse;
import ru.yandex.direct.core.entity.moderation.model.ModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationResult;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.ModerationResponseProcessingOp;

public abstract class BaseResponseOperationsChain
        <T extends AbstractModerationResultResponse<? extends ModerationMeta, ? extends ModerationResult>>
        implements ResponseOperationsChain<T> {

    private final ApplicationContext applicationContext;

    protected BaseResponseOperationsChain(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    private List<ModerationResponseProcessingOp<T>> getOperations() {
        return getOperationClasses().stream()
                .sorted(Comparator.comparing(Class::getName))
                .map(this::lookupBean)
                .collect(Collectors.toList());
    }

    private ModerationResponseProcessingOp<T> lookupBean(Class<? extends ModerationResponseProcessingOp<T>> opClass) {
        return applicationContext.getBean(opClass);
    }

    @Override
    public void accept(Configuration cfg, BulkUpdateHolder bulkUpdateHolder, List<T> objects) {
        if (objects.isEmpty()) {
            return;
        }

        var operations = getOperations();

        for (var op : operations) {
            for (var object : objects) {
                if (restrictedModeEnabled(object)) {
                    // В ограниченном режиме ничего с принятым вердиктом не делаем
                    continue;
                }
                op.consume(bulkUpdateHolder, object);
            }
        }

        for (var op : operations) {
            op.flush(cfg, bulkUpdateHolder);
        }

        bulkUpdateHolder.execute(cfg);
    }

    /**
     * Возвращает флаг, показывающий, в каком режиме работает транспорт для конкретного типа объектов -
     * ограниченном или обычном.
     *
     * Ограниченный транспорт нужен в процессе разработки сверхновой модерации
     * (см. https://st.yandex-team.ru/DIRECT-130974), потом нужно включать обычный (полноценный).
     *
     * @return false - транспорт работает в обычном режиме, true - в ограниченном
     */
    protected boolean restrictedModeEnabled(T response) {
        return false;
    }
}
