package ru.yandex.direct.core.entity.moderation.service.receiving.processor;

import java.util.List;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;

import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResponse;
import ru.yandex.direct.core.entity.moderation.model.ModerationMeta;

import static com.google.common.base.Preconditions.checkArgument;

public class ModerationResponseParserImpl<T extends AbstractModerationResponse<? extends ModerationMeta>> implements ModerationResponseParser<T, Long> {
    private final Function<T, Long> idGetter;
    private final Function<T, Long> versionGetter;
    private final Function<T, List<Long>> reasonsGetter;
    private Function<T, Long> unixtimeGetter;

    private final Predicate<T> isValid;
    private final BiPredicate<T, List<Long>> hasCriticalReason;

    private ModerationResponseParserImpl(Function<T, Long> idGetter, Function<T, Long> versionGetter,
                                         Function<T, List<Long>> reasonsGetter, Function<T, Long> unixtimeGetter,
                                         Predicate<T> isValid, BiPredicate<T, List<Long>> hasCriticalReason) {

        this.idGetter = idGetter;
        this.versionGetter = versionGetter;
        this.reasonsGetter = reasonsGetter;
        this.unixtimeGetter = unixtimeGetter;
        this.isValid = isValid;
        this.hasCriticalReason = hasCriticalReason;
    }

    public static <G extends AbstractModerationResponse<? extends ModerationMeta>> ModerationResponseParserImplBuilder<G> builder() {
        return new ModerationResponseParserImplBuilder<>();
    }

    @Override
    public Long getId(T response) {
        return idGetter.apply(response);
    }

    @Override
    public Long getVersion(T response) {
        return versionGetter.apply(response);
    }

    @Override
    public List<Long> getReasons(T response) {
        return reasonsGetter.apply(response);
    }

    @Override
    public Long getUnixtime(T response) {
        return unixtimeGetter.apply(response);
    }

    @Override
    public boolean isValid(T response) {
        return isValid.test(response);
    }

    @Override
    public boolean hasCriticalReason(T response) {
        return hasCriticalReason.test(response, getReasons(response));
    }

    @Override
    public Long getKey(T response) {
        return getId(response);
    }

    public static final class ModerationResponseParserImplBuilder<T extends AbstractModerationResponse<?
            extends ModerationMeta>> {
        private Function<T, Long> idGetter;
        private Function<T, Long> versionGetter;
        private Function<T, List<Long>> reasonsGetter;
        private Function<T, Long> unixtimeGetter;
        private Predicate<T> isValid;
        private BiPredicate<T, List<Long>> hasCriticalReason;

        private ModerationResponseParserImplBuilder() {
        }


        public ModerationResponseParserImplBuilder<T> withIdGetter(Function<T, Long> idGetter) {
            this.idGetter = idGetter;
            return this;
        }

        public ModerationResponseParserImplBuilder<T> withVersionGetter(Function<T, Long> versionGetter) {
            this.versionGetter = versionGetter;
            return this;
        }

        public ModerationResponseParserImplBuilder<T> withReasonsGetter(Function<T, List<Long>> reasonsGetter) {
            this.reasonsGetter = reasonsGetter;
            return this;
        }

        public ModerationResponseParserImplBuilder<T> withUnixtimeGetter(Function<T, Long> unixtimeGetter) {
            this.unixtimeGetter = unixtimeGetter;
            return this;
        }

        public ModerationResponseParserImplBuilder<T> withIsValid(Predicate<T> isValid) {
            this.isValid = isValid;
            return this;
        }

        public ModerationResponseParserImplBuilder<T> withHasCriticalReason(BiPredicate<T, List<Long>> hasCriticalReason) {
            this.hasCriticalReason = hasCriticalReason;
            return this;
        }

        public ModerationResponseParserImpl<T> build() {
            checkArgument(idGetter != null);
            checkArgument(versionGetter != null);
            checkArgument(reasonsGetter != null);
            checkArgument(unixtimeGetter != null);
            checkArgument(isValid != null);
            checkArgument(hasCriticalReason != null);

            return new ModerationResponseParserImpl<>(idGetter, versionGetter, reasonsGetter, unixtimeGetter, isValid,
                    hasCriticalReason);
        }
    }
}
