package ru.yandex.direct.core.entity.moderation.service.receiving.processor;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import org.jooq.Configuration;

import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResultResponse;
import ru.yandex.direct.core.entity.moderation.model.ModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationResult;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.BaseModerationReceivingService.ModeratedObjectKeyWithVersion;
import ru.yandex.direct.core.entity.moderation.service.receiving.processing_configurations.ResponseOperationsChain;
import ru.yandex.direct.core.entity.moderation.service.sending.ObjectTypeInitialVersionUtil;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.moderation.service.receiving.BaseModerationReceivingService.ModeratedObjectKeyWithVersion.ANY_VERSION;

public class ModerationResponsesProcessingMachine<T extends AbstractModerationResultResponse<? extends ModerationMeta,
        ? extends ModerationResult>, U extends Comparable<U>> {
    private final ModerateResponseChooser moderateResponseChooser;
    private final ModerationResponseParser<T, U> responseParser;

    private final ResponseOperationsChain<T> operationsHolder;

    private final BiFunction<Configuration, List<ModeratedObjectKeyWithVersion<U>>, List<U>> objectLocker;
    private final BiConsumer<List<T>, Configuration> responsesConsumer;
    private final BulkUpdateHolder bulkUpdateHolder;

    private ModerationResponsesProcessingMachine(ModerateResponseChooser moderateResponseChooser,
                                                 ModerationResponseParser<T, U> responseParser,
                                                 ResponseOperationsChain<T> operationsHolder,
                                                 BiFunction<Configuration, List<ModeratedObjectKeyWithVersion<U>>,
                                                         List<U>> objectLocker,
                                                 BiConsumer<List<T>, Configuration> responsesConsumer,
                                                 BulkUpdateHolder bulkUpdateHolder) {
        this.moderateResponseChooser = moderateResponseChooser;
        this.responseParser = responseParser;
        this.operationsHolder = operationsHolder;
        this.objectLocker = objectLocker;
        this.responsesConsumer = responsesConsumer;
        this.bulkUpdateHolder = bulkUpdateHolder;
    }

    public ModerationResponseProcessingResult<T> processResponses(List<T> responses, Configuration configuration) {
        int[] unknownVerdictCount = new int[1];
        unknownVerdictCount[0] = responses.size();

        Map<U, T> responsesById = responses
                .stream()
                .filter(responseParser::isValid)
                .peek(e -> unknownVerdictCount[0]--)
                .collect(groupingBy(responseParser::getKey, Collectors.collectingAndThen(toList(),
                        el -> moderateResponseChooser.apply(el, responseParser))));

        // Сортируем объекты с вердиктами, чтоб порядок объектов при попытке взять лок
        // был всегда одинаковый относительно отправки запросов и приемки вердиктов.
        List<ModeratedObjectKeyWithVersion<U>> objectIdWithVersions = responsesById.entrySet()
                .stream()
                .map(e -> wrapResponse(e.getKey(), e.getValue()))
                .sorted(Comparator.comparing(ModeratedObjectKeyWithVersion::getKey))
                .collect(toList());

        List<U> locked = objectLocker.apply(configuration, objectIdWithVersions);

        List<T> lockedResponses = locked.stream().map(responsesById::get).collect(toList());

        responsesConsumer.accept(lockedResponses, configuration);

        operationsHolder.accept(configuration, bulkUpdateHolder, lockedResponses);

        bulkUpdateHolder.execute(configuration);

        return new ModerationResponseProcessingResult<>(lockedResponses, unknownVerdictCount[0]);
    }

    private ModeratedObjectKeyWithVersion<U> wrapResponse(U id, T response) {
        long actualVersion = responseParser.getVersion(response);

        // Если версия в вердикте меньше, чем версия с которой может отправлять объект новый транспорт,
        // то это вердикт на смигрированную версию и его нужно применять в любом случае.
        boolean applyToAllVersions = actualVersion < ObjectTypeInitialVersionUtil.getInitialVersion(response.getType());

        return new ModeratedObjectKeyWithVersion(id, applyToAllVersions ? ANY_VERSION : actualVersion);
    }

    public static <G extends AbstractModerationResultResponse<? extends ModerationMeta, ? extends ModerationResult>, U extends Comparable<U>>
    ModerationResponsesConsumerBuilder<G, U> builder() {
        return new ModerationResponsesConsumerBuilder<>();
    }

    public static final class ModerationResponsesConsumerBuilder<T extends AbstractModerationResultResponse<?
            extends ModerationMeta, ? extends ModerationResult>, U extends Comparable<U>> {
        private BiFunction<Configuration, List<ModeratedObjectKeyWithVersion<U>>, List<U>> objectLocker;
        private BiConsumer<List<T>, Configuration> responsesConsumer;
        private ResponseOperationsChain<T> responseProcessingOps;
        private ModerationResponseParser<T, U> responseParser;
        private ModerateResponseChooser moderateResponseChooser;
        private BulkUpdateHolder bulkUpdateHolder;

        private ModerationResponsesConsumerBuilder() {
        }

        public ModerationResponsesConsumerBuilder<T, U> withModerateResponseChooser(ModerateResponseChooser moderateResponseChooser) {
            this.moderateResponseChooser = moderateResponseChooser;
            return this;
        }

        public ModerationResponsesConsumerBuilder<T, U> withObjectLocker(BiFunction<Configuration,
                List<ModeratedObjectKeyWithVersion<U>>, List<U>> objectLocker) {
            this.objectLocker = objectLocker;
            return this;
        }

        public ModerationResponsesConsumerBuilder<T, U> withResponsesConsumer(BiConsumer<List<T>, Configuration> responsesConsumer) {
            this.responsesConsumer = responsesConsumer;
            return this;
        }

        public ModerationResponsesConsumerBuilder<T, U> withOperations(ResponseOperationsChain<T> processingChainBuilderGetter) {
            this.responseProcessingOps = processingChainBuilderGetter;
            return this;
        }

        public ModerationResponsesConsumerBuilder<T, U> withResponseParser(ModerationResponseParser<T, U> responseParser) {
            this.responseParser = responseParser;
            return this;
        }

        public ModerationResponsesConsumerBuilder<T, U> withBulkUpdateHolder(BulkUpdateHolder bulkUpdateHolder) {
            this.bulkUpdateHolder = bulkUpdateHolder;
            return this;
        }

        public ModerationResponsesProcessingMachine<T, U> build() {
            checkArgument(responseParser != null);
            checkArgument(responseProcessingOps != null);
            checkArgument(objectLocker != null);
            checkArgument(responsesConsumer != null);
            checkArgument(moderateResponseChooser != null);
            checkArgument(bulkUpdateHolder != null);

            return new ModerationResponsesProcessingMachine<T, U>(
                    moderateResponseChooser, responseParser, responseProcessingOps, objectLocker,
                    responsesConsumer, bulkUpdateHolder);
        }
    }
}
