package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.Collection;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdImageModerationInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow;
import ru.yandex.direct.core.entity.moderation.model.ad_image.AdImageBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.ad_image.AdImageBannerRequestData;
import ru.yandex.direct.core.entity.moderation.repository.sending.AbstractAdImageBannerModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.BusinessUnitModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.service.sending.banner.ModerationFlagsConverter;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.BannersSendingContext;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.AUTO_ACCEPT;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.COMMON;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.MANUAL;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.AD_IMAGE;
import static ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames.DIRECT_SERVICE;

public abstract class AbstractAdImageBannerSender extends BaseBannerSender<AdImageBannerModerationRequest,
        BannerWithAdImageModerationInfo, BannerModerationMeta> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractAdImageBannerSender.class);

    public static final long INITIAL_VERSION = 75_000L;

    private final GeoTreeFactory geoTreeFactory;
    private final TurboLandingRepository turboLandingRepository;

    public AbstractAdImageBannerSender(DslContextProvider dslContextProvider,
                                       AbstractAdImageBannerModerationRepository moderationSendingRepository,
                                       RemoderationFlagsRepository remoderationFlagsRepository,
                                       CampaignRepository campaignRepository, GeoTreeFactory geoTreeFactory,
                                       HrefParameterizingService hrefParameterizingService,
                                       TurboLandingRepository turboLandingRepository, FeatureService featureService,
                                       BusinessUnitModerationRepository businessUnitModerationRepository,
                                       ModerationFlagsConverter moderationFlagsConverter,
                                       ModerationOperationModeProvider moderationOperationModeProvider) {
        super(dslContextProvider, moderationSendingRepository, remoderationFlagsRepository, campaignRepository,
                hrefParameterizingService, featureService, businessUnitModerationRepository,
                moderationFlagsConverter, moderationOperationModeProvider);
        this.geoTreeFactory = geoTreeFactory;
        this.turboLandingRepository = turboLandingRepository;
    }

    @Override
    protected BannerModerationMeta makeMetaObject() {
        return new BannerModerationMeta();
    }

    @Override
    public String typeName() {
        return "ad_image";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected AdImageBannerModerationRequest convertBanner(BannerWithAdImageModerationInfo moderationInfo,
                                                           long version) {
        AdImageBannerModerationRequest request = new AdImageBannerModerationRequest();

        BannersSendingContext bannersSendingContext = (BannersSendingContext) getContext();

        request.setService(DIRECT_SERVICE);
        request.setType(AD_IMAGE);

        BannerModerationMeta meta = createMetaFromBanner(moderationInfo);
        meta.setVersionId(version);

        AdImageBannerRequestData data = new AdImageBannerRequestData();

        data.setDomain(moderationInfo.getDomain());
        data.setBody(moderationInfo.getBody());
        data.setTitle(moderationInfo.getTitle());
        data.setTitleExtension(moderationInfo.getTitleExtension());
        data.setHref(moderationInfo.getHref());
        data.setGeo(StreamEx.of(getGeoTree().getModerationCountries(moderationInfo.getGeo())).joining(","));
        data.setHost(moderationInfo.getHost());
        data.setImageHash(moderationInfo.getImageHash());
        data.setMdsGroupId(moderationInfo.getMdsGroupId());
        data.setNamespace(moderationInfo.getNamespace());

        if (moderationInfo.getHref() != null) {
            data.setParametrizedHref(bannersSendingContext.getParameterizedUrl(moderationInfo.getHref()));
        }

        request.setMeta(meta);
        request.setData(data);

        return request;
    }

    @Override
    public long initialVersion() {
        return INITIAL_VERSION;
    }

    @Override
    protected ModerationWorkflow getWorkflow(BannerWithAdImageModerationInfo moderationInfo) {
        if (moderationInfo.getBidReModerate() != null) {
            return MANUAL;
        } else if (moderationInfo.getBidAutoModerate() != null) {
            return AUTO_ACCEPT;
        } else {
            return COMMON;
        }
    }

    @Override
    protected void postProcess(Configuration configuration,
                               Collection<BannerWithAdImageModerationInfo> idsForSendStatus) {
        super.postProcess(configuration, idsForSendStatus);
        // Переотправляем турбу на модерацию, подробности - https://st.yandex-team.ru/DIRECT-117291
        turboLandingRepository.resendTurboLandingToModeration(configuration,
                idsForSendStatus.stream().map(Banner::getId).collect(Collectors.toList()));
    }

    private GeoTree getGeoTree() {
        // При отправке в модерацию баннеров всех типов нужно использовать российское гео дерево (DIRECTSUP-33318)
        return geoTreeFactory.getRussianGeoTree();
    }
}
