package ru.yandex.direct.core.entity.moderation.service.sending

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.banner.model.BannerMulticardSetWithModerationInfo
import ru.yandex.direct.core.entity.banner.model.BannerMulticardWithModerationInfo
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.client.model.ClientFlags
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider
import ru.yandex.direct.core.entity.moderation.model.asset.BannerAssetModerationMeta
import ru.yandex.direct.core.entity.moderation.model.asset.BannerMulticardModerationRequest
import ru.yandex.direct.core.entity.moderation.model.asset.BannerMulticardRequestData
import ru.yandex.direct.core.entity.moderation.model.asset.MulticardModerationBannerType
import ru.yandex.direct.core.entity.moderation.model.asset.MulticardModerationCpmVideoBanner
import ru.yandex.direct.core.entity.moderation.model.asset.MulticardModerationDataItem
import ru.yandex.direct.core.entity.moderation.model.asset.MulticardModerationImage
import ru.yandex.direct.core.entity.moderation.model.asset.MulticardModerationTextBanner
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerMulticardSetSendingRepository
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.AutoAcceptanceType
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationType
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.regions.GeoTreeFactory

@ModerationSender
class BannerMulticardSender(
    dslContextProvider: DslContextProvider,
    bannerMulticardSetSendingRepository: BannerMulticardSetSendingRepository,
    moderationOperationModeProvider: ModerationOperationModeProvider,
    remoderationFlagsRepository: RemoderationFlagsRepository,
    campaignRepository: CampaignRepository,
    val geoTreeFactory: GeoTreeFactory,
) : BaseAssetSender<Long, BannerMulticardModerationRequest, BannerMulticardSetWithModerationInfo>(
    dslContextProvider,
    bannerMulticardSetSendingRepository,
    moderationOperationModeProvider,
    remoderationFlagsRepository,
    campaignRepository,
    RemoderationType.MULTICARD,
    AutoAcceptanceType.MULTICARD
) {
    companion object {
        const val initialVersion = 1L
    }

    private val logger = LoggerFactory.getLogger(BannerMulticardSender::class.java)

    // При отправке в модерацию баннеров всех типов нужно использовать российское гео дерево (DIRECTSUP-33318)
    private val geoTree
        get() = geoTreeFactory.russianGeoTree

    override fun typeName(): String = "multicard"

    override fun getLogger(): Logger = logger

    override fun convert(
        moderationInfo: BannerMulticardSetWithModerationInfo,
        version: Long
    ): BannerMulticardModerationRequest {
        val multicardBannerType = MulticardModerationBannerType.fromBannerType(moderationInfo.bannerType)

        val multicards = moderationInfo.multicards
            .map { multicard -> toMulticardModerationDataItem(multicard) }
            .sortedBy { it.order }

        val geo = geoTree.getModerationCountries(moderationInfo.geo).joinToString(",")

        return BannerMulticardModerationRequest().apply {
            service = ModerationServiceNames.DIRECT_SERVICE
            type = ModerationObjectType.BANNER_MULTICARD
            meta = BannerAssetModerationMeta().apply {
                clientId = moderationInfo.clientId
                uid = moderationInfo.uid

                campaignId = moderationInfo.campaignId
                adGroupId = moderationInfo.adGroupId
                bannerId = moderationInfo.bid

                versionId = version
            }

            data = when (multicardBannerType) {
                MulticardModerationBannerType.TEXT -> BannerMulticardRequestData(
                    multicard = multicards,
                    bannerType = multicardBannerType,
                    geo = geo,
                    textBanner = MulticardModerationTextBanner(
                        body = moderationInfo.body,
                        title = moderationInfo.title,
                        titleExtension = moderationInfo.titleExtension,
                        href = moderationInfo.href,
                        domain = moderationInfo.domain,
                    )
                )
                MulticardModerationBannerType.CPM_VIDEO -> BannerMulticardRequestData(
                    multicard = multicards,
                    bannerType = multicardBannerType,
                    geo = geo,
                    cpmVideoBanner = MulticardModerationCpmVideoBanner(
                        href = moderationInfo.href,
                        domain = moderationInfo.domain,
                        creativeId = moderationInfo.creativeId,
                        creativePreviewUrl = moderationInfo.creativePreviewUrl,
                        livePreviewUrl = moderationInfo.livePreviewUrl,
                    )
                )
            }

            if (moderationInfo.clientFlags.contains(ClientFlags.AS_SOON_AS_POSSIBLE.typedValue)) {
                data.asSoonAsPossible = true
            }
        }
    }

    private fun toMulticardModerationDataItem(multicard: BannerMulticardWithModerationInfo) =
        MulticardModerationDataItem(
            id = multicard.multicardId,
            order = multicard.order,
            image = MulticardModerationImage(
                host = multicard.avatarsHost.literal,
                imageHash = multicard.imageHash,
                namespace = multicard.namespace.literal,
                mdsGroupId = multicard.mdsGroupId,
            ),
            text = multicard.text,
            href = multicard.href,
        )

    override fun getVersion(obj: BannerMulticardSetWithModerationInfo): Long {
        return if (obj.version != null) {
            return obj.version + 1
        } else {
            initialVersion
        }
    }
}
