package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.Collection;
import java.util.List;

import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.model.VideoWithModerationInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.creative.model.ModerationInfo;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow;
import ru.yandex.direct.core.entity.moderation.model.asset.BannerVideoAdditionModerationData;
import ru.yandex.direct.core.entity.moderation.model.asset.BannerVideoAdditionModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.asset.BannerVideoAdditionModerationRequest;
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerVideoAdditionSendingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.AUTO_ACCEPT;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.COMMON;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.MANUAL;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.BANNER_VIDEO_ADDITION;
import static ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames.DIRECT_SERVICE;
import static ru.yandex.direct.feature.FeatureName.SOCIAL_ADVERTISING;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ModerationSender
public class BannerVideoAdditionSender extends ModerationSendingService<Long, BannerVideoAdditionModerationRequest,
        VideoWithModerationInfo> {

    public static final long INITIAL_VERSION = 80_000L;

    private static final Logger logger = LoggerFactory.getLogger(BannerVideoAdditionSender.class);

    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final CampaignRepository campaignRepository;
    private final FeatureService featureService;

    public BannerVideoAdditionSender(DslContextProvider dslContextProvider,
                                     BannerVideoAdditionSendingRepository moderationSendingRepository,
                                     ModerationOperationModeProvider moderationOperationModeProvider,
                                     CampaignRepository campaignRepository,
                                     FeatureService featureService) {
        super(dslContextProvider, moderationSendingRepository);
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
    }

    @Override
    public String typeName() {
        return "banner-video-addition";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected BannerVideoAdditionModerationRequest convert(VideoWithModerationInfo moderationInfo,
                                                           long version) {
        BannerVideoAdditionModerationData data = new BannerVideoAdditionModerationData();
        data.setPreviewUrl(moderationInfo.getPreviewUrl());
        data.setLivePreviewUrl(moderationInfo.getLivePreviewUrl());
        data.setDuration(moderationInfo.getDuration());
        data.setModerationInfo(sanitizeModerationInfo(moderationInfo.getModerationInfo()));
        var clientFlags = moderationInfo.getClientFlags();
        if (clientFlags != null && clientFlags.contains(ClientFlags.AS_SOON_AS_POSSIBLE.getTypedValue())) {
            data.setAsSoonAsPossible(true);
        }
        ClientId clientId = ClientId.fromLong(moderationInfo.getClientId());
        boolean isSocialAdvertisingEnabled = featureService.isEnabledForClientId(clientId, SOCIAL_ADVERTISING);
        data.setIsSocialAdvertisement(isSocialAdvertisingEnabled ? Boolean.TRUE : null);

        BannerVideoAdditionModerationMeta meta = new BannerVideoAdditionModerationMeta();
        meta.setBannerId(moderationInfo.getBid());
        meta.setCreativeId(moderationInfo.getCreativeId());
        meta.setAdGroupId(moderationInfo.getAdGroupId());
        meta.setCampaignId(moderationInfo.getCampaignId());
        meta.setClientId(moderationInfo.getClientId());
        meta.setUid(moderationInfo.getUid());
        meta.setVersionId(version);

        BannerVideoAdditionModerationRequest request = new BannerVideoAdditionModerationRequest();
        request.setService(DIRECT_SERVICE);
        request.setType(BANNER_VIDEO_ADDITION);
        request.setMeta(meta);
        request.setData(data);

        return request;
    }

    private ModerationInfo sanitizeModerationInfo(ModerationInfo moderationInfo) {
        if (moderationInfo.getTexts() == null) {
            moderationInfo.setTexts(emptyList());
        }
        if (moderationInfo.getSounds() == null) {
            moderationInfo.setSounds(emptyList());
        }
        return moderationInfo;
    }

    @Override
    protected ModerationWorkflow getWorkflow(VideoWithModerationInfo moderationInfo) {
        if (moderationInfo.getBidReModerate() != null) {
            return MANUAL;
        } else if (moderationInfo.getBidAutoModerate() != null) {
            return AUTO_ACCEPT;
        } else {
            return COMMON;
        }
    }

    @Override
    protected void postProcess(Configuration configuration, Collection<VideoWithModerationInfo> objects) {
        // в ограниченном режиме пропускаем пост-процессинг
        if (moderationOperationModeProvider.getMode(BANNER_VIDEO_ADDITION).equals(RESTRICTED)) {
            return;
        }
        List<Long> campaignIds = mapList(objects, VideoWithModerationInfo::getCampaignId);
        campaignRepository.markReadyCampaignsAsSent(configuration, campaignIds);
    }

    @Override
    protected long getVersion(VideoWithModerationInfo object) {
        if (object.getVersion() != null) {
            return Math.max(INITIAL_VERSION, object.getVersion() + 1);
        } else {
            return INITIAL_VERSION;
        }
    }
}
