package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.jooq.Configuration;

import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.ModerationableBanner;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.BaseBannerModerationData;
import ru.yandex.direct.core.entity.moderation.repository.sending.BusinessUnitModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.ModerationSendingRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.AutoAcceptanceType;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationType;
import ru.yandex.direct.core.entity.moderation.service.sending.banner.ModerationFlagsConverter;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.BannersSendingContext;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.implementations.ReplacingParamsImpl;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.AGE;
import static ru.yandex.direct.core.entity.banner.model.BannerFlags.PLUS18;
import static ru.yandex.direct.core.entity.banner.type.flags.BannerWithFlagsConstraints.ALLOWED_FLAGS;
import static ru.yandex.direct.core.entity.bs.common.service.BsBannerIdCalculator.calculateBsBannerId;
import static ru.yandex.direct.feature.FeatureName.SOCIAL_ADVERTISING;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public abstract class BaseBannerSender<D extends BannerModerationRequest<K, ? extends BaseBannerModerationData>,
        T extends ModerationableBanner, K extends BannerModerationMeta>
        extends ModerationSendingService<Long, D, T> {

    public static final long DEFAULT_INITIAL_VERSION = 1L;

    private static final Set<String> USER_FLAGS = Sets.union(ALLOWED_FLAGS, Set.of(PLUS18.getKey()));

    private final RemoderationFlagsRepository remoderationFlagsRepository;
    private final CampaignRepository campaignRepository;
    private final HrefParameterizingService hrefParameterizingService;
    private final FeatureService featureService;
    private final BusinessUnitModerationRepository businessUnitModerationRepository;
    private final ModerationFlagsConverter moderationFlagsConverter;
    private final ModerationOperationModeProvider moderationOperationModeProvider;

    public BaseBannerSender(DslContextProvider dslContextProvider,
                            ModerationSendingRepository<Long, T> moderationSendingRepository,
                            RemoderationFlagsRepository remoderationFlagsRepository,
                            CampaignRepository campaignRepository,
                            HrefParameterizingService hrefParameterizingService, FeatureService featureService,
                            BusinessUnitModerationRepository businessUnitModerationRepository,
                            ModerationFlagsConverter moderationFlagsConverter,
                            ModerationOperationModeProvider moderationOperationModeProvider) {
        super(dslContextProvider, moderationSendingRepository);
        this.remoderationFlagsRepository = remoderationFlagsRepository;
        this.campaignRepository = campaignRepository;
        this.hrefParameterizingService = hrefParameterizingService;
        this.featureService = featureService;
        this.businessUnitModerationRepository = businessUnitModerationRepository;
        this.moderationFlagsConverter = moderationFlagsConverter;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    protected ModerationOperationModeProvider getModerationOperationModeProvider() {
        return moderationOperationModeProvider;
    }

    @Override
    protected SendingContext makeNewContext(int shard, List<T> objects) {
        Map<String, String> processedUrls = hrefParameterizingService.getOriginalUrlToProcessed(
                shard,
                objects,
                info -> ReplacingParamsImpl.builder()
                        .withCampaignType(info.getCampaignType())
                        .withPid(info.getAdGroupId())
                        .withBid(info.getId())
                        .withCreativeId(null)
                        .withCid(info.getCampaignId())
                        .build(),
                this::extractHrefs);
        List<Long> bids = mapList(objects, ModerationableBanner::getId);

        return new BannersSendingContext(
                processedUrls,
                businessUnitModerationRepository.getBidsToBusinessUnits(shard, bids)
        );
    }

    protected List<String> extractHrefs(T object) {
        return object.getHref() == null ? emptyList() : List.of(object.getHref());
    }

    protected abstract D convertBanner(T moderationInfo, long version);

    @Override
    protected final D convert(T moderationInfo, long version) {
        D convertedBanner = convertBanner(moderationInfo, version);

        setSocialAdvertisement(moderationInfo, convertedBanner);
        setBusinessUnit(moderationInfo, convertedBanner);
        setAsSoonAsPossible(moderationInfo, convertedBanner);
        setBsBannerIs(moderationInfo, convertedBanner);

        setUserFlags(moderationInfo, convertedBanner);

        return convertedBanner;
    }

    private void setSocialAdvertisement(T moderationInfo, D convertedBanner) {
        ClientId clientId = ClientId.fromLong(moderationInfo.getClientId());
        boolean isSocialAdvertisingEnabled = featureService.isEnabledForClientId(clientId, SOCIAL_ADVERTISING);
        convertedBanner.getData().setIsSocialAdvertisement(isSocialAdvertisingEnabled ? Boolean.TRUE : null);
    }

    private void setBusinessUnit(T moderationInfo, D convertedBanner) {
        var context = (BannersSendingContext) getContext();
        convertedBanner.getData()
                .setBusinessUnit(context.getBidToBusinessUnit().get(moderationInfo.getId()));
    }

    private void setAsSoonAsPossible(T moderationInfo, D convertedBanner) {
        if (moderationInfo.getClientFlags().contains(ClientFlags.AS_SOON_AS_POSSIBLE.getTypedValue())) {
            convertedBanner.getData().setAsSoonAsPossible(true);
        }
    }

    private void setUserFlags(T moderationInfo, D convertedBanner) {
        // Костыль. В базе могут быть одновременно флаги age:<x> и plus18,
        // а клиент может редактировать только первый, и в результате они начинают конфликтовать.
        BannerFlags bannerFlags = moderationInfo.getFlags();
        if (bannerFlags != null &&
                bannerFlags.getFlags().containsKey(PLUS18.getKey()) &&
                bannerFlags.getFlags().containsKey(AGE.getKey())) {
            bannerFlags.getFlags().remove(PLUS18.getKey());
        }

        List<Integer> userFlags = moderationFlagsConverter.convertFlags(bannerFlags, USER_FLAGS);
        convertedBanner.getData().setUserFlags(userFlags);
    }

    private void setBsBannerIs(T moderationInfo, D convertedBanner) {
        long bsId;

        if (moderationInfo.getBsBannerId() == 0) {
            bsId = calculateBsBannerId(moderationInfo.getId());
        } else {
            bsId = moderationInfo.getBsBannerId();
        }

        convertedBanner.getMeta().setBsBannerId(bsId);
    }

    protected void setDirectTypes(T moderationInfo, K requestMeta) {
        BannersBannerType bannerType = moderationInfo.getBannerType();
        CampaignType campaignType = moderationInfo.getCampaignType();
        CampaignsType campaignsType = campaignType != null ? CampaignType.toSource(campaignType) : null;
        PhrasesAdgroupType adgroupType = moderationInfo.getAdGroupType();

        if (bannerType != null && campaignsType != null && adgroupType != null) {
            requestMeta.setBannerType(bannerType);
            requestMeta.setCampaignType(campaignsType);
            requestMeta.setAdgroupType(adgroupType);
        }
    }

    @Override
    protected void postProcess(Configuration configuration, Collection<T> idsForSendStatus) {
        // в ограниченном режиме пропускаем пост-процессинг
        if (moderationOperationModeProvider.isForcedRestrictedMode()) {
            return;
        }
        List<Long> bids = idsForSendStatus.stream().map(BannerWithSystemFields::getId).collect(Collectors.toList());
        remoderationFlagsRepository.removeRemoderationFlag(configuration, bids, RemoderationType.BANNER);
        remoderationFlagsRepository.removeAutoAcceptFlag(configuration, bids, AutoAcceptanceType.BANNER);
        campaignRepository.markReadyCampaignsAsSent(configuration, mapList(idsForSendStatus,
                BannerWithSystemFields::getCampaignId));
    }

    /**
     * В зависимости от текущей версии баннера, добавляет запись на добавление или обновление в соответсвующий список.
     * Возвращает версию баннера, которую отправляем на модерацию.
     */
    @Override
    protected long getVersion(ModerationableBanner object) {
        if (moderationOperationModeProvider.isImmutableVersionMode()) {
            return nvl(object.getVersion(), initialVersion());
        }

        if (object.getVersion() != null) {
            return getNextVersion(object.getVersion());
        } else {
            return initialVersion();
        }
    }

    protected abstract K makeMetaObject();

    protected long getNextVersion(long currentVersion) {
        return currentVersion + 1;
    }

    public long initialVersion() {
        return DEFAULT_INITIAL_VERSION;
    }

    protected K createMetaFromBanner(T banner) {
        K meta = makeMetaObject();
        meta.setUid(banner.getUid());
        meta.setClientId(banner.getClientId());
        meta.setCampaignId(banner.getCampaignId());
        meta.setAdGroupId(banner.getAdGroupId());
        meta.setBannerId(banner.getId());
        return meta;
    }

}
