package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.function.Predicate;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.VideoFormat;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.BaseBannerModerationData;
import ru.yandex.direct.core.entity.moderation.repository.sending.BusinessUnitModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.ModerationSendingRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.service.sending.banner.ModerationFlagsConverter;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;

public abstract class BaseVideoBannerSender<T extends BannerModerationRequest<BannerModerationMeta,
        ? extends BaseBannerModerationData>, D extends BannerWithModerationInfo>
        extends BaseBannerSender<T, D, BannerModerationMeta> {
    public static final int DEFAULT_WIDTH = 1440;
    public static final int DEFAULT_HEIGHT = 720;
    private static final Logger logger = LoggerFactory.getLogger(BaseVideoBannerSender.class);

    public BaseVideoBannerSender(DslContextProvider dslContextProvider,
                                 ModerationSendingRepository<Long, D> moderationSendingRepository,
                                 RemoderationFlagsRepository remoderationFlagsRepository,
                                 CampaignRepository campaignRepository,
                                 HrefParameterizingService hrefParameterizingService, FeatureService featureService,
                                 BusinessUnitModerationRepository businessUnitModerationRepository,
                                 ModerationFlagsConverter moderationFlagsConverter,
                                 ModerationOperationModeProvider moderationOperationModeProvider) {
        super(dslContextProvider, moderationSendingRepository, remoderationFlagsRepository, campaignRepository,
                hrefParameterizingService, featureService, businessUnitModerationRepository, moderationFlagsConverter,
                moderationOperationModeProvider);
    }

    @Override
    protected BannerModerationMeta makeMetaObject() {
        return new BannerModerationMeta();
    }

    @Nullable
    protected VideoFormat getVideoFormatWithLargestWidth(BannerWithModerationInfo banner) {
        List<VideoFormat> formats = getBannerFormats(banner);
        if (isEmpty(formats)) {
            return null;
        }

        return formats.stream()
                .filter(filterVideoFormatMp4())
                .filter(filterVideoFormatHasWidthAndHeight(banner))
                .max(comparatorLargestWidthVideoFormat())
                .orElse(null);
    }

    protected Comparator<VideoFormat> comparatorLargestWidthVideoFormat() {
        return Comparator.comparingInt(VideoFormat::getWidth);
    }

    protected int[] size(@Nullable VideoFormat format) {
        return format == null
                ? new int[]{DEFAULT_WIDTH, DEFAULT_HEIGHT}
                : new int[]{format.getWidth(), format.getHeight()};
    }

    // защищаемся от NPE, чтобы не свалить весь транспорт
    protected String getVideoUrlSafe(BannerWithModerationInfo banner, @Nullable VideoFormat format) {
        if (format == null) {
            logger.error("no video url in banner {}", banner.getId());
            return null;
        }
        return format.getUrl();
    }

    protected Predicate<VideoFormat> filterVideoFormatMp4() {
        return format -> Objects.equals("video/mp4", format.getType());
    }

    protected Predicate<VideoFormat> filterVideoFormatHasWidthAndHeight(BannerWithModerationInfo banner) {
        return format -> {
            if (format.getWidth() == null || format.getHeight() == null) {
                logger.warn("Banner {} has video/mp4 creative format with null width/height", banner.getId());
                return false;
            }
            return true;
        };
    }

    protected List<VideoFormat> getBannerFormats(BannerWithModerationInfo banner) {
        AdditionalData additionalData = banner.getAdditionalData();
        if (additionalData == null) {
            return emptyList();
        }

        List<VideoFormat> formats = additionalData.getFormats();
        if (isEmpty(formats)) {
            return emptyList();
        }

        return formats;
    }

}
