package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithModerationInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.campaign.CampaignModerationData;
import ru.yandex.direct.core.entity.moderation.model.campaign.CampaignModerationMetaImpl;
import ru.yandex.direct.core.entity.moderation.model.campaign.CampaignModerationRequest;
import ru.yandex.direct.core.entity.moderation.repository.sending.CampaignSendingRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.MOD_EXPORT_CAMPAIGNS_ONLY;
import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Ready;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Sent;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.CAMPAIGN;
import static ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames.DIRECT_SERVICE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ModerationSender
public class CampaignSender extends ModerationSendingService<Long, CampaignModerationRequest,
        CampaignWithModerationInfo> {
    private static final Logger logger = LoggerFactory.getLogger(CampaignSender.class);

    public final static long INITIAL_VERSION = 10L;

    private final CampaignRepository campaignRepository;
    private final ModerationOperationModeProvider moderationOperationModeProvider;

    @Autowired
    public CampaignSender(DslContextProvider dslContextProvider,
                          CampaignRepository campaignRepository,
                          CampaignSendingRepository campaignSendingRepository,
                          ModerationOperationModeProvider moderationOperationModeProvider) {
        super(dslContextProvider, campaignSendingRepository);
        this.campaignRepository = campaignRepository;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    @Override
    public String typeName() {
        return "campaign";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected CampaignModerationRequest convert(CampaignWithModerationInfo moderationInfo, long version) {
        CampaignModerationData data = new CampaignModerationData();
        data.setEmail(moderationInfo.getEmail());
        data.setFio(moderationInfo.getFio());
        data.setLogin(moderationInfo.getLogin());
        data.setName(moderationInfo.getName());
        var strategyData = moderationInfo.getStrategyData();
        if (strategyData != null) {
            data.setPayForConversion(strategyData.getPayForConversion());
        }
        var clientFlags = moderationInfo.getClientFlags();
        data.setSpellingAutoCorrect(clientFlags == null || !clientFlags.contains(ClientFlags.NO_TEXT_AUTOCORRECTION.getTypedValue()));

        CampaignModerationMetaImpl meta = new CampaignModerationMetaImpl();
        meta.setCampaignId(moderationInfo.getId());
        meta.setClientId(moderationInfo.getClientId());
        meta.setUid(moderationInfo.getUid());
        meta.setVersionId(version);

        CampaignModerationRequest request = new CampaignModerationRequest();
        request.setService(DIRECT_SERVICE);
        request.setType(CAMPAIGN);
        request.setWorkflow(getWorkflow(moderationInfo));
        request.setData(data);
        request.setMeta(meta);

        return request;
    }

    @Override
    protected void postProcess(Configuration configuration, Collection<CampaignWithModerationInfo> objects) {
    }

    @Override
    protected long getVersion(CampaignWithModerationInfo campaignWithModerationInfo) {
        if (campaignWithModerationInfo.getVersion() != null) {
            return campaignWithModerationInfo.getVersion() + 1;
        } else {
            return INITIAL_VERSION;
        }
    }

    @Override
    protected void updateMysqlDataBeforeSending(Configuration configuration,
                                                List<Pair<CampaignWithModerationInfo, Long>> updatedVersions) {
        // У кампаний нет statusModerate=Sending, не переводим их в этот статус.
        moderationSendingRepository.setModerationVersions(configuration, updatedVersions);
    }

    @Override
    protected void afterSendTransaction(int shard, AtomicReference<List<CampaignWithModerationInfo>> objects) {
        if (moderationOperationModeProvider.getMode(CAMPAIGN).equals(RESTRICTED)) {
            return;
        }

        dslContextProvider.ppcTransaction(shard, configuration -> {
            Collection<CampaignWithModerationInfo> campaigns = objects.get();

            // У кампаний нет statusModerate=Sending, из Ready сразу переводим в Sent или Yes.

            Map<Boolean, List<CampaignWithModerationInfo>> campaignsByModExportType = campaigns.stream()
                    .collect(Collectors.partitioningBy(e -> MOD_EXPORT_CAMPAIGNS_ONLY.contains(e.getType())));

            // Некоторые типы кампаний сразу помечаем, как принятые.
            // https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Moderate/Export.pm?rev=r9189257#L1502
            var autoAcceptedCampaigns = campaignsByModExportType.get(Boolean.TRUE);
            if (!autoAcceptedCampaigns.isEmpty()) {
                var ids = mapList(autoAcceptedCampaigns, e -> e.getId());
                campaignRepository.markCampaignsAsModerated(shard, ids);
                campaignRepository.updateStatusBsSyncedWithLastChange(configuration, ids, StatusBsSynced.NO);
            }

            if (!campaignsByModExportType.get(Boolean.FALSE).isEmpty()) {
                moderationSendingRepository.updateStatusModerate(configuration, Ready, Sent,
                        campaignsByModExportType.get(Boolean.FALSE));
            }
        });
    }
}
