package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.AspectRatio;
import ru.yandex.direct.core.entity.moderation.model.BannerLink;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow;
import ru.yandex.direct.core.entity.moderation.model.cpm.canvas.CanvasBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.canvas.CanvasBannerRequestData;
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerWithCreativesModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.BusinessUnitModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames;
import ru.yandex.direct.core.entity.moderation.service.sending.banner.ModerationFlagsConverter;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.BannersSendingContext;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.AUTO_ACCEPT;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.COMMON;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.MANUAL;
import static ru.yandex.direct.core.entity.moderation.model.cpm.canvas.CanvasModerationInfoConverter.toCanvasModerationInfo;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.CANVAS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class CanvasBannerSender extends BaseBannerSender<CanvasBannerModerationRequest, BannerWithModerationInfo,
        BannerModerationMeta> {
    private static final Logger logger = LoggerFactory.getLogger(CanvasBannerSender.class);

    public static final long INITIAL_VERSION = 5000L;

    private final GeoTreeFactory geoTreeFactory;
    private final TurboLandingRepository turboLandingRepository;

    public CanvasBannerSender(DslContextProvider dslContextProvider,
                              BannerWithCreativesModerationRepository bannerWithCreativesModerationRepository,
                              GeoTreeFactory geoTreeFactory,
                              RemoderationFlagsRepository remoderationFlagsRepository,
                              CampaignRepository campaignRepository,
                              HrefParameterizingService hrefParameterizingService,
                              TurboLandingRepository turboLandingRepository,
                              ModerationOperationModeProvider moderationOperationModeProvider,
                              FeatureService featureService,
                              BusinessUnitModerationRepository businessUnitModerationRepository,
                              ModerationFlagsConverter moderationFlagsConverter) {
        super(dslContextProvider, bannerWithCreativesModerationRepository, remoderationFlagsRepository,
                campaignRepository, hrefParameterizingService, featureService, businessUnitModerationRepository,
                moderationFlagsConverter, moderationOperationModeProvider);

        this.geoTreeFactory = geoTreeFactory;
        this.turboLandingRepository = turboLandingRepository;
    }

    @Override
    protected long getNextVersion(long currentVersion) {
        return Math.max(INITIAL_VERSION, currentVersion + 1);
    }

    @Override
    public long initialVersion() {
        return INITIAL_VERSION;
    }

    @Override
    protected BannerModerationMeta makeMetaObject() {
        return new BannerModerationMeta();
    }

    @Override
    public String typeName() {
        return "canvas_banner";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected CanvasBannerModerationRequest convertBanner(BannerWithModerationInfo moderationInfo, long version) {
        CanvasBannerModerationRequest request = new CanvasBannerModerationRequest();

        BannersSendingContext bannersSendingContext = (BannersSendingContext) getContext();

        request.setService(ModerationServiceNames.DIRECT_SERVICE);
        request.setType(CANVAS);

        BannerModerationMeta meta = createMetaFromBanner(moderationInfo);
        meta.setVersionId(version);
        setDirectTypes(moderationInfo, meta);

        CanvasBannerRequestData data = new CanvasBannerRequestData();

        data.setDomain(moderationInfo.getDomain());
        data.setCreativeId(moderationInfo.getCreativeId());
        data.setLinks(extractBannerLinks(moderationInfo));
        data.setHref(moderationInfo.getHref());
        data.setPreviewUrl(moderationInfo.getCreativePreviewUrl());
        data.setGeo(StreamEx.of(getGeoTree().getModerationCountries(moderationInfo.getGeo())).joining(","));
        data.setAspectRatio(new AspectRatio(moderationInfo.getWidth(), moderationInfo.getHeight()));

        if (moderationInfo.getHref() != null) {
            data.setParametrizedHref(bannersSendingContext.getParameterizedUrl(moderationInfo.getHref()));
        }

        //todo copy moderationInfo (?)
        data.setModerationInfo(toCanvasModerationInfo(moderationInfo.getModerateInfo()));

        request.setMeta(meta);
        request.setData(data);

        return request;
    }

    private List<BannerLink> extractBannerLinks(BannerWithModerationInfo moderationInfo) {
        BannersSendingContext bannersSendingContext = (BannersSendingContext) getContext();
        List<BannerLink> links = new ArrayList<>();

        if (moderationInfo.getHref() != null || moderationInfo.getTurbolandingHref() != null) {
            links.add(createBannerLink(bannersSendingContext, moderationInfo.getHref(),
                    moderationInfo.getTurbolandingHref()));
        }

        for (var additionalHref : moderationInfo.getAdditionalHrefs()) {
            links.add(createBannerLink(bannersSendingContext, additionalHref.getHref(), null));
        }

        return links;
    }

    private BannerLink createBannerLink(BannersSendingContext bannersSendingContext, String href,
                                        @Nullable String turbolandingHref) {
        var link = new BannerLink().setHref(href).setParametrizedHref(bannersSendingContext.getParameterizedUrl(href));
        if (turbolandingHref != null) {
            link.setMobileHref(turbolandingHref);
        }
        return link;
    }

    @Override
    protected ModerationWorkflow getWorkflow(BannerWithModerationInfo moderationInfo) {
        if (moderationInfo.getBidReModerate() != null) {
            return MANUAL;
        } else if (moderationInfo.getBidAutoModerate() != null) {
            return AUTO_ACCEPT;
        } else {
            return COMMON;
        }
    }

    @Override
    protected void postProcess(Configuration configuration, Collection<BannerWithModerationInfo> idsForSendStatus) {
        // в ограниченном режиме пропускаем пост-процессинг
        if (getModerationOperationModeProvider().getMode(CANVAS).equals(RESTRICTED)) {
            return;
        }

        super.postProcess(configuration, idsForSendStatus);

        // Переотправляем турбу на модерацию, подробности - https://st.yandex-team.ru/DIRECT-117291
        turboLandingRepository.resendTurboLandingToModeration(configuration, mapList(idsForSendStatus, Banner::getId));
    }

    private GeoTree getGeoTree() {
        // При отправке в модерацию баннеров всех типов нужно использовать российское гео дерево (DIRECTSUP-33318)
        return geoTreeFactory.getRussianGeoTree();
    }
}
