package ru.yandex.direct.core.entity.moderation.service.sending;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.displayhrefs.model.DisplayHrefWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.displayhrefs.DisplayHrefsModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.displayhrefs.DisplayHrefsModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.displayhrefs.DisplayHrefsRequestData;
import ru.yandex.direct.core.entity.moderation.repository.sending.DisplayHrefsSendingRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.AutoAcceptanceType;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationType;
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.queryrec.model.Language;

import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.DISPLAYHREFS;

@ModerationSender
public class DisplayHrefsSender
        extends BaseAssetSender<Long, DisplayHrefsModerationRequest, DisplayHrefWithModerationInfo> {
    private static final Logger logger = LoggerFactory.getLogger(DisplayHrefsSender.class);

    public final static long VERSION_OFFSET = 30000L;

    private final QueryrecService queryrecService;

    @Autowired
    public DisplayHrefsSender(DslContextProvider dslContextProvider,
                              DisplayHrefsSendingRepository displayHrefsSendingRepository,
                              ModerationOperationModeProvider moderationOperationModeProvider,
                              RemoderationFlagsRepository remoderationFlagsRepository,
                              CampaignRepository campaignRepository,
                              QueryrecService queryrecService) {
        super(dslContextProvider, displayHrefsSendingRepository, moderationOperationModeProvider,
                remoderationFlagsRepository, campaignRepository,
                RemoderationType.DISPLAY_HREFS, AutoAcceptanceType.DISPLAY_HREFS);
        this.queryrecService = queryrecService;
    }

    @Override
    public String typeName() {
        return "displayhrefs";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected DisplayHrefsModerationRequest convert(DisplayHrefWithModerationInfo moderationInfo,
                                                    long version) {
        DisplayHrefsModerationRequest request = new DisplayHrefsModerationRequest();
        request.setService(ModerationServiceNames.DIRECT_SERVICE);
        request.setType(DISPLAYHREFS);

        DisplayHrefsModerationMeta displayhrefsModerationMeta = new DisplayHrefsModerationMeta();
        displayhrefsModerationMeta.setUid(moderationInfo.getUid());
        displayhrefsModerationMeta.setDisplayHrefId(moderationInfo.getId());
        displayhrefsModerationMeta.setVersionId(version);
        displayhrefsModerationMeta.setClientId(moderationInfo.getClientId());
        displayhrefsModerationMeta.setCampaignId(moderationInfo.getCampaignId());
        displayhrefsModerationMeta.setBannerId(moderationInfo.getBid());
        displayhrefsModerationMeta.setAdGroupId(moderationInfo.getAdGroupId());

        DisplayHrefsRequestData displayHrefsRequestData = new DisplayHrefsRequestData();
        displayHrefsRequestData.setText(moderationInfo.getText());

        Language language = queryrecService.recognize(moderationInfo.getText(),
                ClientId.fromLong(moderationInfo.getClientId()), null);

        String langName = language == Language.UNKNOWN ? null : language.getName();

        displayHrefsRequestData.setLang(langName);

        if (moderationInfo.getClientFlags().contains(ClientFlags.AS_SOON_AS_POSSIBLE.getTypedValue())) {
            displayHrefsRequestData.setAsSoonAsPossible(true);
        }

        request.setMeta(displayhrefsModerationMeta);
        request.setData(displayHrefsRequestData);

        return request;
    }

    @Override
    protected long getVersion(DisplayHrefWithModerationInfo object) {
        if (object.getVersion() != null) {
            return object.getVersion() + 1;
        } else {
            return VERSION_OFFSET;
        }
    }
}
