package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerTurboLandingStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.banner.type.turbolanding.BannerTurboLandingRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.model.ModerationInfo;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoImage;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.AspectRatio;
import ru.yandex.direct.core.entity.moderation.model.BannerLink;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.cpm.frontpage.FixcpmYndxFrontpageBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.frontpage.FixcpmYndxFrontpageBannerRequestData;
import ru.yandex.direct.core.entity.moderation.repository.sending.BusinessUnitModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.CpmFrontpageBannersRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames;
import ru.yandex.direct.core.entity.moderation.service.sending.banner.ModerationFlagsConverter;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.FIXCPM_YNDX_FRONTPAGE_CREATIVE;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@ModerationSender
public class FixcpmFrontpageBannerSender extends BaseBannerSender<FixcpmYndxFrontpageBannerModerationRequest,
        BannerWithModerationInfo, BannerModerationMeta> {

    public static final int CPM_PRICE_EXPANDED_MIN_WIDTH = 640;
    public static final int CPM_PRICE_EXPANDED_MAX_WIDTH = 1200;
    public static final int CPM_PRICE_EXPANDED_MIN_HEIGHT = 360;
    public static final int CPM_PRICE_EXPANDED_MAX_HEIGHT = 675;


    private static final Logger logger = LoggerFactory.getLogger(FixcpmFrontpageBannerSender.class);

    private final GeoTreeFactory geoTreeFactory;
    private final BannerTurboLandingRepository bannerTurbolandingRepository;

    @Autowired
    public FixcpmFrontpageBannerSender(DslContextProvider dslContextProvider,
                                       GeoTreeFactory geoTreeFactory,
                                       BannerTurboLandingRepository bannerTurbolandingRepository,
                                       CpmFrontpageBannersRepository cpmFrontpageBannersRepository,
                                       RemoderationFlagsRepository remoderationFlagsRepository,
                                       CampaignRepository campaignRepository,
                                       HrefParameterizingService hrefParameterizingService,
                                       FeatureService featureService,
                                       BusinessUnitModerationRepository businessUnitModerationRepository,
                                       ModerationFlagsConverter moderationFlagsConverter,
                                       ModerationOperationModeProvider moderationOperationModeProvider) {
        super(dslContextProvider, cpmFrontpageBannersRepository, remoderationFlagsRepository, campaignRepository,
                hrefParameterizingService, featureService, businessUnitModerationRepository,
                moderationFlagsConverter, moderationOperationModeProvider);
        this.geoTreeFactory = geoTreeFactory;

        this.bannerTurbolandingRepository = bannerTurbolandingRepository;
    }

    @Override
    protected BannerModerationMeta makeMetaObject() {
        return new BannerModerationMeta();
    }

    @Override
    protected void postProcess(Configuration configuration, Collection<BannerWithModerationInfo> banners) {
        super.postProcess(configuration, banners);

        bannerTurbolandingRepository.updateBannerTurbolandingsStatusModerate(configuration,
                listToMap(
                        banners.stream().map(Banner::getId).collect(Collectors.toList()),
                        identity(),
                        v -> BannerTurboLandingStatusModerate.SENT));
    }

    @Override
    public String typeName() {
        return "fixcpm_frontpage_banner";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected FixcpmYndxFrontpageBannerModerationRequest convertBanner(BannerWithModerationInfo moderationInfo,
                                                                       long version) {
        FixcpmYndxFrontpageBannerModerationRequest request = new FixcpmYndxFrontpageBannerModerationRequest();

        request.setService(ModerationServiceNames.DIRECT_SERVICE);
        request.setType(FIXCPM_YNDX_FRONTPAGE_CREATIVE);

        BannerModerationMeta meta = createMetaFromBanner(moderationInfo);
        meta.setVersionId(version);

        FixcpmYndxFrontpageBannerRequestData data = new FixcpmYndxFrontpageBannerRequestData();

        data.setDomain(moderationInfo.getDomain());
        data.setLinks(extractBannerLinks(moderationInfo));
        data.setLogin(moderationInfo.getLogin());
        data.setCreativeId(moderationInfo.getCreativeId());
        data.setCreativePreviewUrl(moderationInfo.getCreativePreviewUrl());
        data.setLivePreviewUrl(moderationInfo.getLivePreviewUrl());
        data.setAspectRatio(new AspectRatio(moderationInfo.getWidth(), moderationInfo.getHeight()));
        data.setSimplePicture(moderationInfo.getSimplePicture());
        data.setGeo(StreamEx.of(getGeoTree().getModerationCountries(moderationInfo.getGeo())).joining(","));
        data.setCurrency(moderationInfo.getCurrency().name());
        data.setArchiveUrl(moderationInfo.getArchiveUrl());
        data.setMinExtendedAspectRatio(new AspectRatio(CPM_PRICE_EXPANDED_MIN_WIDTH, CPM_PRICE_EXPANDED_MIN_HEIGHT));
        data.setMaxExtendedAspectRatio(new AspectRatio(CPM_PRICE_EXPANDED_MAX_WIDTH, CPM_PRICE_EXPANDED_MAX_HEIGHT));
        data.setExtendedPreviewUrl(moderationInfo.getExpandedPreviewUrl());
        data.setVideoDuration(moderationInfo.getDuration());
        if (moderationInfo.getModerateInfo() != null) {
            data.setVideoUrl(getVideoUrl(moderationInfo.getModerateInfo()));
            data.setBgrcolor(moderationInfo.getModerateInfo().getBgrcolor());
            data.setCreativePreviewUrl(getCreativePreviewUrl(moderationInfo.getModerateInfo()));
        }

        request.setMeta(meta);
        request.setData(data);

        return request;
    }

    private String getCreativePreviewUrl(ModerationInfo moderationInfo) {
        if (moderationInfo.getImages() == null) {
            return null;
        }
        return moderationInfo.getImages().stream()
                .filter(it -> it.getType() != null && it.getType().equals("screenshot"))
                .map(ModerationInfoImage::getUrl)
                .findAny().orElse(null);
    }

    private String getVideoUrl(ModerationInfo moderationInfo) {
        if (moderationInfo.getVideos() == null || moderationInfo.getVideos().isEmpty()) {
            return null;
        }
        return moderationInfo.getVideos().get(0).getUrl();
    }

    private List<BannerLink> extractBannerLinks(BannerWithModerationInfo moderationInfo) {
        List<BannerLink> links = new ArrayList<>();
        links.add(new BannerLink()
                .setHref(moderationInfo.getHref())
                .setMobileHref(moderationInfo.getTurbolandingHref()));

        for (BannerAdditionalHref additionalHref : moderationInfo.getAdditionalHrefs()) {
            links.add(new BannerLink().setHref(additionalHref.getHref()));
        }

        return links;
    }

    private GeoTree getGeoTree() {
        // При отправке в модерацию баннеров всех типов нужно использовать российское гео дерево (DIRECTSUP-33318)
        return geoTreeFactory.getRussianGeoTree();
    }
}
