package ru.yandex.direct.core.entity.moderation.service.sending;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.ImageWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.asset.BannerAssetModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.image.ImageModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.image.ImageRequestData;
import ru.yandex.direct.core.entity.moderation.repository.sending.ImageSendingRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.AutoAcceptanceType;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationType;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames.DIRECT_SERVICE;

@ModerationSender
public class ImageSender extends BaseAssetSender<Long, ImageModerationRequest,
        ImageWithModerationInfo> {
    private static final Logger logger = LoggerFactory.getLogger(ImageSender.class);

    public final static long VERSION_OFFSET = 10_000_000L;

    @Autowired
    public ImageSender(DslContextProvider dslContextProvider,
                       ImageSendingRepository imageSendingRepository,
                       ModerationOperationModeProvider moderationOperationModeProvider,
                       RemoderationFlagsRepository remoderationFlagsRepository,
                       CampaignRepository campaignRepository) {
        super(dslContextProvider, imageSendingRepository, moderationOperationModeProvider,
                remoderationFlagsRepository, campaignRepository,
                RemoderationType.BANNER_IMAGE, AutoAcceptanceType.BANNER_IMAGE);
    }

    @Override
    public String typeName() {
        return "images";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected ImageModerationRequest convert(ImageWithModerationInfo moderationInfo,
                                             long version) {
        ImageModerationRequest request = new ImageModerationRequest();
        request.setService(DIRECT_SERVICE);
        request.setType(ModerationObjectType.IMAGES);

        BannerAssetModerationMeta meta = new BannerAssetModerationMeta();
        meta.setUid(moderationInfo.getUid());
        meta.setVersionId(version);
        meta.setClientId(moderationInfo.getClientId());
        meta.setCampaignId(moderationInfo.getCampaignId());
        meta.setBannerId(moderationInfo.getBid());
        meta.setAdGroupId(moderationInfo.getAdGroupId());

        ImageRequestData imageRequestData = new ImageRequestData();
        imageRequestData.setImageHash(moderationInfo.getImageHash());

        if (moderationInfo.getMdsGroupId() != null) {
            imageRequestData.setHost(AvatarHost.toSource(moderationInfo.getAvatarsHost()).getLiteral());
            imageRequestData.setMdsGroupId(moderationInfo.getMdsGroupId());
            imageRequestData.setNamespace(BannerImageFormatNamespace.toSource(moderationInfo.getNamespace()).getLiteral());
        } else {
            imageRequestData.setName(moderationInfo.getName());
        }

        if (moderationInfo.getClientFlags().contains(ClientFlags.AS_SOON_AS_POSSIBLE.getTypedValue())) {
            imageRequestData.setAsSoonAsPossible(true);
        }

        CampaignsType campaignsType = moderationInfo.getCampaignType() != null
                ? CampaignType.toSource(moderationInfo.getCampaignType()) : null;
        if (campaignsType != null) {
            imageRequestData.setCampaignType(campaignsType);
        }

        request.setMeta(meta);
        request.setData(imageRequestData);

        return request;
    }

    @Override
    protected long getVersion(ImageWithModerationInfo object) {
        if (object.getVersion() != null) {
            return object.getVersion() + 1;
        } else {
            return VERSION_OFFSET;
        }
    }
}
