package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.model.BannerWithInternalAdModerationInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.internalad.InternalBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.repository.sending.BusinessUnitModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.InternalBannerModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.service.sending.banner.ModerationFlagsConverter;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.core.entity.moderation.service.sending.internalad.InternalBannerModerationHelper;
import ru.yandex.direct.core.entity.moderation.service.sending.internalad.InternalBannersSendingContext;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.internalads.Constants.MODERATED_TEMPLATES_RESOURCES_MAPPER;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.INTERNAL_BANNER;
import static ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames.DIRECT_SERVICE;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;

@ModerationSender
@ParametersAreNonnullByDefault
public class InternalBannerSender extends BaseBannerSender<InternalBannerModerationRequest,
        BannerWithInternalAdModerationInfo, BannerModerationMeta> {

    private static final Logger logger = LoggerFactory.getLogger(InternalBannerSender.class);


    private final GeoTreeFactory geoTreeFactory;
    private final BannerImageFormatRepository bannerImageFormatRepository;

    @Autowired
    public InternalBannerSender(DslContextProvider dslContextProvider,
                                InternalBannerModerationRepository moderationSendingRepository,
                                RemoderationFlagsRepository remoderationFlagsRepository,
                                CampaignRepository campaignRepository,
                                GeoTreeFactory geoTreeFactory,
                                HrefParameterizingService hrefParameterizingService,
                                FeatureService featureService,
                                BusinessUnitModerationRepository businessUnitModerationRepository,
                                BannerImageFormatRepository bannerImageFormatRepository,
                                ModerationFlagsConverter moderationFlagsConverter,
                                ModerationOperationModeProvider moderationOperationModeProvider) {
        super(dslContextProvider, moderationSendingRepository, remoderationFlagsRepository, campaignRepository,
                hrefParameterizingService, featureService, businessUnitModerationRepository,
                moderationFlagsConverter, moderationOperationModeProvider);
        this.geoTreeFactory = geoTreeFactory;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
    }

    @Override
    protected BannerModerationMeta makeMetaObject() {
        return new BannerModerationMeta();
    }

    @Override
    public String typeName() {
        return "internal_banner";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected SendingContext makeNewContext(int shard, List<BannerWithInternalAdModerationInfo> objects) {
        Set<String> imageHashes = flatMapToSet(objects, InternalBannerSender::getBannerImageHashes);

        Map<String, BannerImageFormat> imageFormatByHash =
                bannerImageFormatRepository.getBannerImageFormats(shard, imageHashes);

        return new InternalBannersSendingContext(imageFormatByHash);
    }

    private static Set<String> getBannerImageHashes(BannerWithInternalAdModerationInfo moderationInfo) {
        var resourceIdToProperty = MODERATED_TEMPLATES_RESOURCES_MAPPER.get(moderationInfo.getTemplateId());
        if (resourceIdToProperty == null) {
            // такое может быть, если пытаемся отправить неизвестный шаблон. При конвертации на это есть checkState
            return Collections.emptySet();
        }

        return StreamEx.of(moderationInfo.getTemplateVariables())
                .mapToEntry(var -> resourceIdToProperty.get(var.getTemplateResourceId()))
                .nonNullValues()
                .filterValues(Constants::isImageProperty)
                .keys()
                .map(TemplateVariable::getInternalValue)
                .nonNull()
                .toSet();
    }

    @Override
    protected InternalBannerModerationRequest convertBanner(BannerWithInternalAdModerationInfo moderationInfo,
                                                            long version) {
        // ожидаем, что статус модерации сбрасывается только у баннеров с шаблонами из
        // MODERATED_TEMPLATES_RESOURCES_MAPPER
        // иначе падаем и чиним баг в том месте где неправильно сбросили статус модерации
        checkState(MODERATED_TEMPLATES_RESOURCES_MAPPER.containsKey(moderationInfo.getTemplateId()),
                "Unexpected templateId %s for moderation internal banner", moderationInfo.getTemplateId());

        checkState(moderationInfo.getModerationInfo() != null,
                "moderationInfo must not be null for banner %s", moderationInfo.getId());

        BannerModerationMeta meta = createMetaFromBanner(moderationInfo);
        meta.setVersionId(version);

        var moderationCountries = StreamEx.of(getGeoTree().getModerationCountries(moderationInfo.getGeo())).joining(",");
        var context = (InternalBannersSendingContext) getContext();
        var data = InternalBannerModerationHelper.INSTANCE
                .createInternalBannerRequestData(moderationInfo, true, context.getImageFormatByHash(),
                        moderationCountries);

        var request = new InternalBannerModerationRequest();
        request.setService(DIRECT_SERVICE);
        request.setType(INTERNAL_BANNER);
        request.setMeta(meta);
        request.setData(data);
        return request;
    }

    private GeoTree getGeoTree() {
        // При отправке в модерацию баннеров всех типов нужно использовать российское гео дерево (DIRECTSUP-33318)
        return geoTreeFactory.getRussianGeoTree();
    }
}
