package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Configuration;
import org.slf4j.Logger;

import ru.yandex.direct.core.entity.moderation.model.Attributes;
import ru.yandex.direct.core.entity.moderation.model.ModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow;
import ru.yandex.direct.core.entity.moderation.model.Moderationable;
import ru.yandex.direct.core.entity.moderation.repository.sending.ModerationSendingRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.joining;
import static org.apache.commons.collections4.CollectionUtils.subtract;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.COMMON;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Ready;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Sending;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Sent;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * @param <U> ключ отправляемого объекта
 * @param <D> тип запроса в модерацию
 * @param <E> объект, отправляемый в модерацию
 */
public abstract class ModerationSendingService<U, D extends ModerationRequest<? extends ModerationMeta>,
        E extends Moderationable> {
    protected final DslContextProvider dslContextProvider;
    protected final ModerationSendingRepository<U, E> moderationSendingRepository;

    public ModerationSendingService(DslContextProvider dslContextProvider,
                                    ModerationSendingRepository<U, E> moderationSendingRepository) {
        this.dslContextProvider = dslContextProvider;
        this.moderationSendingRepository = moderationSendingRepository;
    }

    public abstract String typeName();

    protected abstract Logger getLogger();

    protected abstract D convert(E moderationInfo, long version);

    protected abstract void postProcess(Configuration configuration, Collection<E> objects);

    protected abstract long getVersion(E object);

    public interface SendingContext {

    }

    private final ThreadLocal<SendingContext> sendingContext = new ThreadLocal<>();

    protected SendingContext makeNewContext(int shard, List<E> objects) {
        return null;
    }

    protected SendingContext getContext() {
        return sendingContext.get();
    }

    protected Collection<U> lockIds(Configuration configuration, Collection<U> ids) {
        Collection<U> lockedIds = moderationSendingRepository.lockKeys(ids, configuration);

        if (lockedIds.isEmpty()) {
            getLogger().warn("Fail to lock {} for moderation! {}",
                    typeName(), ids.stream().map(Object::toString).collect(joining(", ")));

            return Set.of();
        } else if (lockedIds.size() != ids.size()) {
            Collection<U> notLockedIds =
                    subtract(ids, new HashSet<>(lockedIds));

            getLogger().warn("Fail to lock {} for moderation! {}",
                    typeName(), notLockedIds.stream().map(Object::toString).collect(joining(", ")));
        }

        return new HashSet<>(lockedIds);
    }

    protected List<E> loadObjectForModeration(Collection<U> lockedIds, Configuration configuration) {
        return moderationSendingRepository.loadObjectForModeration(lockedIds, configuration);
    }

    public void send(int shard, Collection<U> ids,
                     Function<E, Long> eventTimeProvider,
                     Function<E, String> essTagProvider,
                     Consumer<List<D>> sender) {
        send(shard, ids, eventTimeProvider, essTagProvider, null, sender);
    }

    public void send(int shard, Collection<U> ids,
                     Function<E, Long> eventTimeProvider,
                     Function<E, String> essTagProvider,
                     Function<E, ModerationWorkflow> workflowProvider,
                     Consumer<List<D>> sender) {

        AtomicReference<List<D>> requests = new AtomicReference<>();
        AtomicReference<List<E>> objects = new AtomicReference<>();

        // Сортируем объекты с вердиктами, чтоб порядок объектов при попытке взять лок
        // был всегда одинаковый относительно отправки запросов и приемки вердиктов.
        var sortedIds = Set.copyOf(ids).stream().sorted().collect(Collectors.toList());
        beforeSendTransaction(shard, sortedIds, objects, requests, eventTimeProvider, essTagProvider,
                workflowProvider);

        if (requests.get() == null || requests.get().isEmpty()) {
            return;
        }

        sender.accept(requests.get());

        afterSendTransaction(shard, objects);
        sendingContext.remove();
    }

    protected void afterSendTransaction(int shard, AtomicReference<List<E>> objects) {
        dslContextProvider.ppcTransaction(shard, configuration -> {
            postProcess(configuration, objects.get());
            moderationSendingRepository.updateStatusModerate(configuration, Sending, Sent, objects.get());
        });
    }

    void beforeSendTransaction(int shard, Collection<U> ids,
                               AtomicReference<List<E>> objects,
                               AtomicReference<List<D>> requests,
                               Function<E, Long> eventTimeProvider,
                               Function<E, String> essTagProvider) {
        beforeSendTransaction(shard, ids, objects, requests, eventTimeProvider, essTagProvider, null);
    }

    void beforeSendTransaction(int shard, Collection<U> ids,
                               AtomicReference<List<E>> objects,
                               AtomicReference<List<D>> requests,
                               Function<E, Long> eventTimeProvider,
                               Function<E, String> essTagProvider,
                               Function<E, ModerationWorkflow> workflowProvider) {

        dslContextProvider.ppcTransaction(shard, configuration -> {
            Collection<U> lockedIds = lockIds(configuration, ids); //  Ready | Sending

            if (lockedIds.isEmpty()) {
                return;
            }

            objects.set(loadObjectForModeration(lockedIds, configuration));

            if (lockedIds.size() != objects.get().size()) {
                // TODO: вместо исключения стали писать в лог в задаче DIRECT-127807.
                // А правильно было бы вместо этого не брать лок на объекты, которые не нужно модерировать.
                getLogger().error("Loaded not all locked objects, locked: {}", lockedIds);
//                throw new IllegalStateException("Loaded not all locked objects, locked: " + lockedIds);
            }

            setNewContext(makeNewContext(shard, objects.get()));

            requests.set(sendObjectsList(objects.get(), eventTimeProvider, essTagProvider, workflowProvider, configuration));
        });
    }

    void setNewContext(SendingContext context) {
        sendingContext.set(context);
    }

    List<D> sendObjectsList(Collection<E> objects,
                            Function<E, Long> eventTimeProvider,
                            Function<E, String> essTagProvider,
                            Function<E, ModerationWorkflow> workflowProvider,
                            Configuration configuration) {
        List<D> requests = new ArrayList<>(objects.size());
        List<Pair<E, Long>> updatedVersions = new ArrayList<>();

        for (var object : objects) {
            long newVersion;

            newVersion = getVersion(object);
            updatedVersions.add(Pair.of(object, newVersion));

            ModerationWorkflow workflow = selectWorkflow(workflowProvider, object);
            D request = convert(object, newVersion);
            request.setWorkflow(workflow);

            fillEventTime(request, eventTimeProvider.apply(object));
            fillEssTag(request, essTagProvider.apply(object));

            requests.add(request);
        }

        updateMysqlDataBeforeSending(configuration, updatedVersions);

        return requests;
    }

    /**
     * Функция производит нужные обновления в mysql-базе перед отправкой в Модерацию
     * Изменяется статус модерации нужных объектов на Sending, объектам в базе присваивается новая версия
     *
     * @param updatedVersions список пар объектов и их новых версий
     */
    protected void updateMysqlDataBeforeSending(Configuration configuration,
                                                List<Pair<E, Long>> updatedVersions) {
        moderationSendingRepository.setModerationVersions(configuration, updatedVersions);
        moderationSendingRepository.updateStatusModerate(configuration, Ready, Sending,
                mapList(updatedVersions, Pair::getLeft));
    }

    ModerationWorkflow selectWorkflow(Function<E, ModerationWorkflow> workflowProvider, E object) {
        if (workflowProvider != null) {
            ModerationWorkflow workflow = workflowProvider.apply(object);
            if (workflow != null) {
                return workflow;
            }
        }
        return getWorkflow(object);
    }

    protected ModerationWorkflow getWorkflow(E moderationInfo) {
        /*
           Пока в модерации нет полноценной поддержки этих воркфлоу.
           https://st.yandex-team.ru/DIRECT-109811#5e271e9ad1dda141bfc5f363
         */

        return COMMON;
    }

    private void fillEventTime(ModerationRequest request, Long eventTime) {
        if (eventTime != null) {
            request.setUnixtime(eventTime);
        } else {
            request.setUnixtime(System.currentTimeMillis());
        }
    }

    protected void fillEssTag(ModerationRequest request, String essTag) {
        if (essTag == null) {
            return;
        }

        Attributes attributes = defaultIfNull(request.getAttributes(), new Attributes());
        attributes.setBetaPort(essTag);
        request.setAttributes(attributes);
    }

}
