package ru.yandex.direct.core.entity.moderation.service.sending

import org.apache.commons.lang3.tuple.Pair
import org.jooq.Configuration
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper
import ru.yandex.direct.core.entity.client.model.ClientFlags
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow
import ru.yandex.direct.core.entity.moderation.model.promoextension.PromoExtensionModerationMeta
import ru.yandex.direct.core.entity.moderation.model.promoextension.PromoExtensionModerationRequest
import ru.yandex.direct.core.entity.moderation.model.promoextension.PromoExtensionModerationRequestData
import ru.yandex.direct.core.entity.moderation.repository.sending.PromoExtensionSendingRepository
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.implementations.ReplacingParamsImpl
import ru.yandex.direct.core.entity.promoextension.model.PromoExtensionWithModerationInfo
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

@ModerationSender
class PromoExtensionSender(
    private val bannersUrlHelper: BannersUrlHelper,
    private val hrefParameterizingService: HrefParameterizingService,
    private val promoExtensionSendingRepository: PromoExtensionSendingRepository,
    dslContextProvider: DslContextProvider,
) : ModerationSendingService<Long, PromoExtensionModerationRequest, PromoExtensionWithModerationInfo>(
    dslContextProvider,
    promoExtensionSendingRepository,
) {
    companion object {
        const val initialVersion = 1L
    }
    private val logger = LoggerFactory.getLogger(PromoExtensionSender::class.java)

    override fun typeName() = "promo_extension"

    override fun getLogger(): Logger = logger

    override fun convert(
            moderationInfo: PromoExtensionWithModerationInfo,
            version: Long
    ): PromoExtensionModerationRequest {
        return PromoExtensionModerationRequest().apply {
            service = ModerationServiceNames.DIRECT_SERVICE
            type = ModerationObjectType.PROMO_EXTENSION
            meta = PromoExtensionModerationMeta(
                moderationInfo.promoExtensionId,
                uid = moderationInfo.uid,
                clientId = moderationInfo.clientId,
                versionId = version,
            )
            data = PromoExtensionModerationRequestData(
                description = moderationInfo.compoundDescription,
                href = moderationInfo.href,
                parametrizedHref = moderationInfo.href?.let { makeParametrizedHref(it, context as PromoactionSendingContext) },
                domain = moderationInfo.href?.let { extractDomain(it) },
                promocode = null,
            )
            if (moderationInfo.clientFlags.contains(ClientFlags.AS_SOON_AS_POSSIBLE.typedValue)) {
                data.asSoonAsPossible = true
            }
        }
    }

    private fun makeParametrizedHref(href: String, sendingContext: PromoactionSendingContext) = sendingContext.parameterizedUrls[href]

    private fun extractDomain(href: String) = bannersUrlHelper.extractHostFromHrefWithWwwOrNull(href)

    override fun getVersion(obj: PromoExtensionWithModerationInfo) = obj.version + 1

    override fun postProcess(configuration: Configuration, objects: Collection<PromoExtensionWithModerationInfo>) {
    }

    override fun updateMysqlDataBeforeSending(
        configuration: Configuration,
        updatedVersions: List<Pair<PromoExtensionWithModerationInfo, Long>>
    ) {
        promoExtensionSendingRepository.updateMysqlDataBeforeSending(configuration, updatedVersions)
    }

    override fun makeNewContext(
            shard: Int,
            objects: List<PromoExtensionWithModerationInfo>,
    ) = PromoactionSendingContext(
        hrefParameterizingService.getOriginalUrlToProcessed(
            shard,
            objects,
            { ReplacingParamsImpl.builder().build() },
            { promo -> promo.href?.let { listOf(it) } ?: listOf() }
        )
    )

    inner class PromoactionSendingContext(val parameterizedUrls: Map<String, String>) : SendingContext
}
