package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.sitelinks.SitelinksModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.sitelinks.SitelinksModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.sitelinks.SitelinksRequestData;
import ru.yandex.direct.core.entity.moderation.repository.sending.SitelinksSendingRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.AutoAcceptanceType;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationType;
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.BannersSendingContext;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.implementations.ReplacingParamsImpl;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSetWithModerationInfo;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkWithTurboHref;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.SITELINKS_SET;

@ModerationSender
public class SitelinksSender
        extends BaseAssetSender<Long, SitelinksModerationRequest, SitelinkSetWithModerationInfo> {

    private static final Logger logger = LoggerFactory.getLogger(SitelinksSender.class);

    public final static long VERSION_OFFSET = 40_000_000L;
    private final HrefParameterizingService hrefParameterizingService;

    @Autowired
    public SitelinksSender(DslContextProvider dslContextProvider,
                           SitelinksSendingRepository sitelinksSendingRepository,
                           ModerationOperationModeProvider moderationOperationModeProvider,
                           RemoderationFlagsRepository remoderationFlagsRepository,
                           CampaignRepository campaignRepository,
                           HrefParameterizingService hrefParameterizingService) {
        super(dslContextProvider, sitelinksSendingRepository, moderationOperationModeProvider,
                remoderationFlagsRepository, campaignRepository,
                RemoderationType.SITELINKS_SET, AutoAcceptanceType.SITELINKS_SET);
        this.hrefParameterizingService = hrefParameterizingService;
    }

    @Override
    public String typeName() {
        return "sitelinks";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected SitelinksModerationRequest convert(SitelinkSetWithModerationInfo moderationInfo,
                                                 long version) {
        SitelinksModerationRequest request = new SitelinksModerationRequest();
        request.setService(ModerationServiceNames.DIRECT_SERVICE);
        request.setType(SITELINKS_SET);

        SitelinksModerationMeta sitelinksModerationMeta = new SitelinksModerationMeta();
        sitelinksModerationMeta.setUid(moderationInfo.getUid());
        sitelinksModerationMeta.setSitelinksSetId(moderationInfo.getId());
        sitelinksModerationMeta.setVersionId(version);
        sitelinksModerationMeta.setClientId(moderationInfo.getClientId());
        sitelinksModerationMeta.setCampaignId(moderationInfo.getCampaignId());
        sitelinksModerationMeta.setBannerId(moderationInfo.getBid());
        sitelinksModerationMeta.setAdGroupId(moderationInfo.getAdGroupId());

        SitelinksRequestData sitelinksRequestData = new SitelinksRequestData();

        sitelinksRequestData.setSitelinkRequestList(convertLinks(moderationInfo.getSitelinks()));

        if (moderationInfo.getClientFlags().contains(ClientFlags.AS_SOON_AS_POSSIBLE.getTypedValue())) {
            sitelinksRequestData.setAsSoonAsPossible(true);
        }

        request.setMeta(sitelinksModerationMeta);
        request.setData(sitelinksRequestData);

        return request;
    }

    private List<SitelinksRequestData.SitelinkRequest> convertLinks(Collection<SitelinkWithTurboHref> sitelinks) {
        List<SitelinksRequestData.SitelinkRequest> sitelinkRequests = new ArrayList<>();

        BannersSendingContext bannersSendingContext = (BannersSendingContext) getContext();

        for (SitelinkWithTurboHref sitelink : sitelinks) {
            SitelinksRequestData.SitelinkRequest r = new SitelinksRequestData.SitelinkRequest();
            r.setTitle(sitelink.getTitle());
            r.setDescription(sitelink.getDescription());
            r.setHref(sitelink.getHref());

            if (sitelink.getHref() != null) {
                r.setParametrizedHref(bannersSendingContext.getParameterizedUrl(sitelink.getHref()));
            }

            r.setSitelinkId(sitelink.getId());
            r.setTurbolinkHref(sitelink.getTurbolandingHref());
            r.setTurbolinkId(sitelink.getTurboLandingId());

            sitelinkRequests.add(r);
        }

        return sitelinkRequests;
    }

    @Override
    protected long getVersion(SitelinkSetWithModerationInfo object) {
        if (object.getVersion() != null) {
            return object.getVersion() + 1;
        } else {
            return VERSION_OFFSET;
        }
    }

    @Override
    protected SendingContext makeNewContext(int shard, List<SitelinkSetWithModerationInfo> objects) {
        Map<String, String> processedUrls = hrefParameterizingService.getOriginalUrlToProcessed(
                shard,
                objects,
                info -> ReplacingParamsImpl.builder()
                        .withCampaignType(info.getCampaignType())
                        .withPid(info.getAdGroupId())
                        .withBid(info.getBid())
                        .withCreativeId(null)
                        .withCid(info.getCampaignId())
                        .build(),
                this::extractHrefs
        );

        return new BannersSendingContext(processedUrls, Collections.emptyMap());
    }

    private List<String> extractHrefs(SitelinkSetWithModerationInfo sitelink) {
        List<String> result = new ArrayList<>();

        for (var link : sitelink.getSitelinks()) {
            if (link.getHref() == null) {
                logger.error("Corrupted sitelink_set bid={}", sitelink.getBid());
                continue;
            }

            result.add(link.getHref());
        }
        return result;
    }
}
