package ru.yandex.direct.core.entity.moderation.service.sending;

import java.util.Collection;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithTextAndImageModerationInfo;
import ru.yandex.direct.core.entity.banner.model.Language;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow;
import ru.yandex.direct.core.entity.moderation.model.asset.ImageModerationData;
import ru.yandex.direct.core.entity.moderation.model.text.TextBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.text.TextBannerRequestData;
import ru.yandex.direct.core.entity.moderation.repository.sending.BusinessUnitModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.TextBannerModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.core.entity.moderation.service.sending.banner.ModerationFlagsConverter;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.BannersSendingContext;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.AUTO_ACCEPT;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.COMMON;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.MANUAL;
import static ru.yandex.direct.core.entity.moderation.service.ModerationObjectType.TEXT_AD;
import static ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames.DIRECT_SERVICE;

@ModerationSender
public class TextBannerSender extends BaseBannerSender<TextBannerModerationRequest, BannerWithTextAndImageModerationInfo,
        BannerModerationMeta> {
    private static final Logger logger = LoggerFactory.getLogger(TextBannerSender.class);

    //TODO INITIAL_VERSION нужно согласовать с модерацией, тикет https://st.yandex-team.ru/BIGMOD-124
    public static final long INITIAL_VERSION = 75_000L;

    private final GeoTreeFactory geoTreeFactory;
    private final BannerRepository bannerRepository;
    private final TurboLandingRepository turboLandingRepository;

    @Autowired
    public TextBannerSender(DslContextProvider dslContextProvider,
                            @Qualifier("textBannerModerationRepository") TextBannerModerationRepository moderationSendingRepository,
                            RemoderationFlagsRepository remoderationFlagsRepository,
                            CampaignRepository campaignRepository,
                            HrefParameterizingService hrefParameterizingService,
                            FeatureService featureService,
                            BusinessUnitModerationRepository businessUnitModerationRepository,
                            GeoTreeFactory geoTreeFactory,
                            BannerRepository bannerRepository,
                            TurboLandingRepository turboLandingRepository,
                            ModerationOperationModeProvider moderationOperationModeProvider,
                            ModerationFlagsConverter moderationFlagsConverter) {
        super(dslContextProvider, moderationSendingRepository, remoderationFlagsRepository, campaignRepository,
                hrefParameterizingService, featureService, businessUnitModerationRepository,
                moderationFlagsConverter, moderationOperationModeProvider);
        this.geoTreeFactory = geoTreeFactory;
        this.bannerRepository = bannerRepository;
        this.turboLandingRepository = turboLandingRepository;
    }

    @Override
    protected BannerModerationMeta makeMetaObject() {
        return new BannerModerationMeta();
    }

    @Override
    public String typeName() {
        return "text_banner";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    protected ModerationObjectType getModerationObjectType() {
        return TEXT_AD;
    }

    @Override
    protected TextBannerModerationRequest convertBanner(BannerWithTextAndImageModerationInfo moderationInfo, long version) {
        TextBannerModerationRequest request = new TextBannerModerationRequest();

        BannersSendingContext bannersSendingContext = (BannersSendingContext) getContext();

        request.setService(DIRECT_SERVICE);
        request.setType(getModerationObjectType());

        BannerModerationMeta meta = createMetaFromBanner(moderationInfo);
        meta.setVersionId(version);
        setDirectTypes(moderationInfo, meta);

        TextBannerRequestData data = new TextBannerRequestData();
        data.setBody(moderationInfo.getBody());
        data.setTitle(moderationInfo.getTitle());
        data.setTitleExtension(moderationInfo.getTitleExtension());
        data.setLanguage(getLanguageAsString(moderationInfo));
        data.setDomain(moderationInfo.getDomain());
        data.setHref(moderationInfo.getHref());
        data.setTurboHref(moderationInfo.getTurboHref());
        data.setGeo(StreamEx.of(getGeoTree().getModerationCountries(moderationInfo.getGeo())).joining(","));
        if (moderationInfo.getHref() != null) {
            data.setParametrizedHref(bannersSendingContext.getParameterizedUrl(moderationInfo.getHref()));
        }
        data.setPermalink(moderationInfo.getPermalink());
        if (moderationInfo.getImageHash() != null && Boolean.TRUE.equals(moderationInfo.getImageStatusShow())) {
            data.addAsset(buildImageData(moderationInfo));
        }
        request.setMeta(meta);
        request.setData(data);

        return request;
    }

    private static ImageModerationData buildImageData(BannerWithTextAndImageModerationInfo moderationInfo) {
        ImageModerationData data = new ImageModerationData();
        data.setImageHash(moderationInfo.getImageHash());
        if (moderationInfo.getImageMdsGroupId() != null) {
            data.setHost(AvatarHost.toSource(moderationInfo.getImageAvatarsHost()).getLiteral());
            data.setMdsGroupId(moderationInfo.getImageMdsGroupId());
            data.setNamespace(BannerImageFormatNamespace.toSource(moderationInfo.getImageNamespace()).getLiteral());
        } else {
            data.setName(moderationInfo.getImageName());
        }
        return data;
    }

    @Override
    protected long getNextVersion(long currentVersion) {
        return Math.max(INITIAL_VERSION, currentVersion + 1);
    }

    @Override
    public long initialVersion() {
        return INITIAL_VERSION;
    }

    @Override
    protected ModerationWorkflow getWorkflow(BannerWithTextAndImageModerationInfo moderationInfo) {
        if (moderationInfo.getBidReModerate() != null) {
            return MANUAL;
        } else if (moderationInfo.getBidAutoModerate() != null) {
            return AUTO_ACCEPT;
        } else {
            return COMMON;
        }
    }

    @Override
    protected void postProcess(Configuration configuration,
                               Collection<BannerWithTextAndImageModerationInfo> idsForSendStatus) {
        // в ограниченном режиме пропускаем пост-процессинг
        if (getModerationOperationModeProvider().getMode(getModerationObjectType()).equals(RESTRICTED)) {
            return;
        }

        super.postProcess(configuration, idsForSendStatus);

        // Переотправляем турбу и отображаемую ссылку на модерацию: https://st.yandex-team.ru/DIRECT-117291
        turboLandingRepository.resendTurboLandingToModeration(configuration,
                idsForSendStatus.stream().map(Banner::getId).collect(Collectors.toList()));

        bannerRepository.displayHrefRepository.resendToModeration(configuration,
                idsForSendStatus.stream().map(Banner::getId).collect(Collectors.toList()));
    }

    // Константа для русского языка называется "RU_", у остальных языков подчеркивания на конце нет
    // https://st.yandex-team.ru/DIRECT-146923
    private static String getLanguageAsString(BannerWithTextAndImageModerationInfo moderationInfo) {
        if (moderationInfo.getLanguage().equals(Language.RU_)) {
            return "RU";
        }
        return moderationInfo.getLanguage().name();
    }

    private GeoTree getGeoTree() {
        // При отправке в модерацию баннеров всех типов нужно использовать российское гео дерево (DIRECTSUP-33318)
        return geoTreeFactory.getRussianGeoTree();
    }
}
