package ru.yandex.direct.core.entity.moderation.service.sending.adgroup;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithModerationInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.keyword.model.KeywordModeration;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.adgroup.AdGroupModerationData;
import ru.yandex.direct.core.entity.moderation.model.adgroup.AdGroupModerationMetaImpl;
import ru.yandex.direct.core.entity.moderation.model.adgroup.AdGroupModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.adgroup.KeywordFields;
import ru.yandex.direct.core.entity.moderation.repository.sending.AdGroupSendingRepository;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames;
import ru.yandex.direct.core.entity.moderation.service.sending.ModerationSender;
import ru.yandex.direct.core.entity.moderation.service.sending.ModerationSendingService;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Sending;
import static ru.yandex.direct.core.entity.moderation.model.TransportStatus.Yes;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ModerationSender
public class AdGroupSender
        extends ModerationSendingService<Long, AdGroupModerationRequest, AdGroupWithModerationInfo> {

    private static final Logger logger = LoggerFactory.getLogger(AdGroupSender.class);

    public static final long INITIAL_VERSION = 30000L;

    private final AdGroupSendingRepository adGroupSendingRepository;
    private final CampaignRepository campaignRepository;
    private final ModerationOperationModeProvider moderationOperationModeProvider;

    @Autowired
    public AdGroupSender(DslContextProvider dslContextProvider,
                         AdGroupSendingRepository adGroupSendingRepository,
                         CampaignRepository campaignRepository,
                         ModerationOperationModeProvider moderationOperationModeProvider) {
        super(dslContextProvider, adGroupSendingRepository);
        this.adGroupSendingRepository = adGroupSendingRepository;
        this.campaignRepository = campaignRepository;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    @Override
    public String typeName() {
        return "adgroup";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected AdGroupModerationRequest convert(AdGroupWithModerationInfo moderationInfo, long version) {
        AdGroupModerationRequest request = new AdGroupModerationRequest();
        request.setService(ModerationServiceNames.DIRECT_SERVICE);
        request.setType(ModerationObjectType.ADGROUP);

        AdGroupModerationMetaImpl meta = new AdGroupModerationMetaImpl();
        meta.setUid(moderationInfo.getUid());
        meta.setClientId(moderationInfo.getClientId());
        meta.setCampaignId(moderationInfo.getCampaignId());
        meta.setAdGroupId(moderationInfo.getId());
        meta.setVersionId(version);

        AdGroupModerationData data = new AdGroupModerationData();
        data.setKeywords(mapList(moderationInfo.getKeywords(), KeywordModeration::getPhrase));
        data.setKeywordsFields(mapList(moderationInfo.getKeywords(),
                kw -> new KeywordFields(kw.getPhrase(), kw.getNormPhrase(), kw.getShowsForecast())));
        if (moderationInfo.getClientFlags().contains(ClientFlags.AS_SOON_AS_POSSIBLE.getTypedValue())) {
            data.setAsSoonAsPossible(true);
        }

        request.setMeta(meta);
        request.setData(data);

        return request;
    }

    @Override
    protected void afterSendTransaction(int shard, AtomicReference<List<AdGroupWithModerationInfo>> objects) {
        if (moderationOperationModeProvider.getMode(ModerationObjectType.ADGROUP).equals(RESTRICTED)) {
            return;
        }
        dslContextProvider.ppcTransaction(shard, configuration -> {
            Collection<AdGroupWithModerationInfo> adGroups = objects.get();
            List<Long> adGroupIds = mapList(adGroups, AdGroupWithModerationInfo::getId);
            postProcess(configuration, adGroups);
            adGroupSendingRepository.updateStatusmoderateAndStatusPostmoderate(configuration, Sending, Yes,
                    PhrasesStatuspostmoderate.Yes, adGroups);
            adGroupSendingRepository.updateKeywordsStatusModerateByAdGroupIds(configuration,
                    Yes, adGroupIds);
        });
    }

    @Override
    protected void postProcess(Configuration configuration, Collection<AdGroupWithModerationInfo> objects) {
        if (moderationOperationModeProvider.getMode(ModerationObjectType.ADGROUP).equals(RESTRICTED)) {
            return;
        }
        // Как и для баннеров и ассетов, переводим соответствующую кампанию из Ready в Sent,
        // т.к., если транспорт кампаний не успеет отправить кампанию в модерацию и проставить статус Sent до пересчета
        // агрегированных статусов, то кампания может не перейти в Yes.
        campaignRepository.markReadyCampaignsAsSent(configuration, mapList(objects,
                AdGroupWithModerationInfo::getCampaignId));
    }

    @Override
    protected long getVersion(AdGroupWithModerationInfo adGroup) {
        if (adGroup.getVersion() != null) {
            return Math.max(INITIAL_VERSION, adGroup.getVersion() + 1);
        } else {
            return INITIAL_VERSION;
        }
    }
}
