package ru.yandex.direct.core.entity.moderation.service.sending.banner;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.modadvert.bigmod.protos.interfaces.MarkupFlags;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
public class ModerationFlagsConverter {

    private static final Logger logger = LoggerFactory.getLogger(ModerationFlagsConverter.class);

    private static final String DEFAULT_VALUE = "1";

    private final ModerationFlagsAliases moderationFlagsAliases;

    /**
     * Мапа строковых представлений флагов во флаги из proto-файла:
     * flag name -> value -> flag
     */
    private final Map<String, Map<String, MarkupFlags.EMarkupFlag>> flagsMap;

    @Autowired
    public ModerationFlagsConverter(ModerationFlagsAliases moderationFlagsAliases) {
        this.moderationFlagsAliases = moderationFlagsAliases;
        this.flagsMap = buildNameToFlagMap();
    }

    private Map<String, Map<String, MarkupFlags.EMarkupFlag>> buildNameToFlagMap() {
        Map<String, Map<String, MarkupFlags.EMarkupFlag>> flagsMap = new HashMap<>();

        StreamEx.of(MarkupFlags.EMarkupFlag.values())
                .forEach(flag -> {
                    MarkupFlags.TFlagProperties properties = flag.getValueDescriptor().getOptions()
                            .getExtension(MarkupFlags.flagProperties);
                    String flagName = properties.getName();
                    String flagValue = properties.getValue();
                    flagsMap.computeIfAbsent(flagName, f -> new HashMap<>()).put(flagValue, flag);
                });

        addNameOverrides(flagsMap);

        ImmutableMap.Builder<String, Map<String, MarkupFlags.EMarkupFlag>> mapBuilder = ImmutableMap.builder();
        flagsMap.forEach((name, valueToFlagMap) -> mapBuilder.put(name, ImmutableMap.copyOf(valueToFlagMap)));
        return mapBuilder.build();
    }

    private void addNameOverrides(Map<String, Map<String, MarkupFlags.EMarkupFlag>> flagsMap) {
        flagsMap.put("plus18", Map.of(DEFAULT_VALUE, MarkupFlags.EMarkupFlag.AGE18));
    }

    @Deprecated
    public List<Integer> convertFlags(Long bannerId, String stringFlags) {
        try {
            return convertFlags(BannerFlags.fromSource(stringFlags), emptySet());
        } catch (Exception e) {
            logger.warn("error while parsing flags for banner_id = {}", bannerId);
            logger.warn(e.getMessage());
            // todo вернуть исключение https://st.yandex-team.ru/DIRECT-153464
//            throw new IllegalStateException(err, e);
            return emptyList();
        }
    }

    public List<Integer> convertFlags(BannerFlags bannerFlags, Set<String> whiteList) {
        if (bannerFlags == null) {
            return emptyList();
        }

        return convertFlags(bannerFlags.getFlags(), whiteList);
    }

    private List<Integer> convertFlags(Map<String, String> flagsMap, Set<String> whiteList) {
        return EntryStream.of(flagsMap)
                .filterKeys(whiteList::contains)
                .mapKeyValue(this::convertToFlagSilently)
                .filter(Objects::nonNull)
                .map(MarkupFlags.EMarkupFlag::getNumber)
                .toList();
    }

    private MarkupFlags.EMarkupFlag convertToFlagSilently(String name, String value) {
        try {
            return convertToFlag(name, value);
        } catch (Exception e) {
            logger.warn(e.getMessage());
            return null;
        }
    }

    private MarkupFlags.EMarkupFlag convertToFlag(String name, String value) {
        if (value == null) {
            value = DEFAULT_VALUE;
        }

        Map<String, MarkupFlags.EMarkupFlag> valueToFlagMap = flagsMap.get(name);
        checkState(valueToFlagMap != null, "can't find flag for name \"%s\"", name);

        String actualValue = nvl(moderationFlagsAliases.getActualValueByAlias(name, value), value);

        MarkupFlags.EMarkupFlag flag = valueToFlagMap.get(actualValue);
        checkState(flag != null, "can't find flag for value \"%s\" (name = \"%s\")", value, name);
        return flag;
    }
}
