package ru.yandex.direct.core.entity.moderation.service.sending.bannerstorage;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.google.common.base.Strings;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.bannerstorage.client.model.File;
import ru.yandex.direct.bannerstorage.client.model.Parameter;
import ru.yandex.direct.bannerstorage.client.model.Template;
import ru.yandex.direct.core.entity.moderation.BannerstorageCreativeWithModerationInfo;
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow;
import ru.yandex.direct.core.entity.moderation.model.bannerstorage.BannerstorageCreativeModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.bannerstorage.BannerstorageCreativeModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.bannerstorage.BannerstorageCreativeParameters;
import ru.yandex.direct.core.entity.moderation.model.bannerstorage.BannerstorageCreativeRequestData;
import ru.yandex.direct.core.entity.moderation.model.bannerstorage.SmartCreativeType;
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerstorageCreativesSendingRepository;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.core.entity.moderation.service.sending.ModerationSendingService;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.bannerstorage.client.BannerStorageClient.SMART_TGO_LAYOUT_ID;
import static ru.yandex.direct.bannerstorage.client.Utils.getSmartCreativeDisplayHeight;
import static ru.yandex.direct.bannerstorage.client.Utils.getSmartCreativeDisplayWidth;
import static ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow.COMMON;
import static ru.yandex.direct.core.entity.moderation.repository.sending.BannerstorageCreativesSendingRepository.SUPPORTED_TEMPLATES;
import static ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames.DIRECT_SERVICE;
import static ru.yandex.direct.utils.JsonUtils.toJson;

public class BannerstorageCreativesSender extends ModerationSendingService<Long, BannerstorageCreativeModerationRequest,
        BannerstorageCreativeWithModerationInfo> {

    public static final long INITIAL_VERSION = 1L;

    private static final Logger logger = LoggerFactory.getLogger(BannerstorageCreativesSender.class);

    private final Map<Integer, Creative> creativesMap;
    private final Map<Integer, Template> templatesMap;
    private final Map<Integer, File> filesMap;
    private final Map<Long, ru.yandex.direct.core.entity.creative.model.Creative> perfCreativesMap;

    public BannerstorageCreativesSender(
            DslContextProvider dslContextProvider,
            BannerstorageCreativesSendingRepository sendingRepository,
            List<Creative> creatives,
            Map<Integer, Template> templatesMap,
            Map<Integer, File> filesMap,
            Map<Long, ru.yandex.direct.core.entity.creative.model.Creative> perfCreativesMap) {
        super(dslContextProvider, sendingRepository);

        this.creativesMap = StreamEx.of(creatives).toMap(Creative::getId, c -> c);
        this.templatesMap = templatesMap;
        this.filesMap = filesMap;
        this.perfCreativesMap = perfCreativesMap;
    }

    @Override
    public String typeName() {
        return "bannerstorage-creative";
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected BannerstorageCreativeModerationRequest convert(BannerstorageCreativeWithModerationInfo moderationInfo,
                                                             long version) {
        var moderationRequest = new BannerstorageCreativeModerationRequest();
        moderationRequest.setService(DIRECT_SERVICE);
        moderationRequest.setType(ModerationObjectType.BANNERSTORAGE_CREATIVES);

        BannerstorageCreativeModerationMeta meta = new BannerstorageCreativeModerationMeta();

        Creative creative = creativesMap.get(moderationInfo.getCreativeId().intValue());

        meta.setClientId(moderationInfo.getClientId());
        meta.setUid(moderationInfo.getUid());
        meta.setCampaignId(moderationInfo.getCampaignId());
        meta.setAdGroupId(moderationInfo.getAdGroupId());
        meta.setBannerId(moderationInfo.getBid());
        meta.setCreativeId(moderationInfo.getCreativeId());
        // Версию креатива указываем ту, которая находится в bannerstorage в статусе "Ожидает модерации"
        // (в противном случае может случиться несовпадение, и вердикт может быть ошибочно применён не к той версии)
        meta.setCreativeVersionId(creative.getVersion());
        meta.setVersionId(version);

        moderationRequest.setMeta(meta);

        moderationRequest.setData(creativeToRequestData(creative, templatesMap, filesMap, perfCreativesMap));

        logger.info("Prepared creative to send: {}", toJson(moderationRequest));
        return moderationRequest;
    }

    private BannerstorageCreativeRequestData creativeToRequestData(
            Creative creative,
            Map<Integer, Template> templatesMap,
            Map<Integer, File> filesMap,
            Map<Long, ru.yandex.direct.core.entity.creative.model.Creative> perfCreativesMap) {

        var data = new BannerstorageCreativeRequestData();

        // url на превью с уточнённой версией креатива (quick & dirty fix для Модерации)
        data.setLivePreviewUrl(creative.getPreview().getUrl() + "&version=" + creative.getVersion());

        Template template = templatesMap.get(creative.getTemplateId());

        // данные шаблона и параметры креатива
        data.setBannerstorageParams(
                new BannerstorageCreativeParameters()
                        .withTemplateId(template.getId())
                        .withTemplateName(SUPPORTED_TEMPLATES.get(template.getId()))
                        .withLogo(extractFileValues(creative, "LOGO", filesMap))
                        .withBackgroundImage(extractFileValues(creative, "BACKGROUND_IMAGE", filesMap))
                        .withDomainList(extractStringValues(creative, "DOMAIN_LIST"))
                        .withUrlSpecialOffer(extractStringValues(creative, "URL_SPECIAL_OFFER"))
                        .withBuyButtonText(extractStringValues(creative, "BUY_BUTTON_TEXT"))
                        .withPrivacyText(extractStringValues(creative, "PRIVACY_TEXT"))
                        .withSpecialOfferPrivacyText(
                                extractStringValues(creative, "SPECIAL_OFFER_PRIVACY_TEXT"))
                        .withSpecialOfferText(extractStringValues(creative, "SPECIAL_OFFER_TEXT"))
                        .withWidth(getDisplayWidth(creative))
                        .withHeight(getDisplayHeight(creative))
                        .withStaticPreview(creative.getScreenshotUrl())
                        .withCreativeType(getSmartCreativeType(creative))
        );

        // geo забираем из perf_creatives
        var perfCreative = perfCreativesMap.get((long) (int) creative.getId());
        var geo = perfCreative.getSumGeo().stream().map(Object::toString).collect(joining(","));
        data.setGeo(geo);

        return data;
    }

    static int getDisplayWidth(Creative creative) {
        return getSmartCreativeDisplayWidth(creative.getWidth(), creative.getLayoutCode().getLayoutId());
    }

    static int getDisplayHeight(Creative creative) {
        return getSmartCreativeDisplayHeight(creative.getHeight(), creative.getLayoutCode().getLayoutId());
    }

    static SmartCreativeType getSmartCreativeType(Creative creative) {
        if (creative.getLayoutCode().getLayoutId() == SMART_TGO_LAYOUT_ID) {
            return SmartCreativeType.TGO;
        }
        if (creative.getWidth() == 0 || creative.getHeight() == 0) {
            return SmartCreativeType.ADAPTIVE;
        }
        return SmartCreativeType.STANDARD;
    }

    private List<String> extractStringValues(Creative creative, String paramName) {
        List<String> values = creative.getParameters().stream()
                .filter(p -> p.getParamName().equals(paramName))
                .map(Parameter::getValues)
                .findFirst()
                .orElse(emptyList());
        return values.stream()
                // frontend в формочках создания/редактирования креативов bannerstorage
                // может добавлять null'ы и пустые строки в некоторых ситуациях
                // модерации такие значения параметров, как правило, не интересны
                .filter(s -> !Strings.isNullOrEmpty(s))
                .collect(toList());
    }

    private List<String> extractFileValues(Creative creative, String paramName, Map<Integer, File> filesMap) {
        List<String> values = creative.getParameters().stream()
                .filter(p -> p.getParamName().equals(paramName))
                .map(Parameter::getValues)
                .findFirst()
                .orElse(emptyList());
        return values.stream()
                // frontend в формочках создания/редактирования креативов bannerstorage
                // может добавлять null'ы и пустые строки в некоторых ситуациях
                // модерации такие значения параметров, как правило, не интересны
                .filter(s -> !Strings.isNullOrEmpty(s))
                // Получаем url файла для указанного fileId
                .map(Integer::parseInt)
                .map(filesMap::get)
                .map(File::getStillageFileUrl)
                .collect(toList());
    }

    @Override
    protected void postProcess(Configuration configuration,
                               Collection<BannerstorageCreativeWithModerationInfo> objects) {
        // Здесь ничего делать не требуется: performance-кампания получает statusModerate=Yes автоматически
        // Возможно, здесь нужно будет добавить простановку статусов для не-performance кампаний
        // (когда будем модерировать не только смарты)
    }

    @Override
    protected long getVersion(BannerstorageCreativeWithModerationInfo object) {
        if (object.getVersion() == null) {
            return INITIAL_VERSION;
        }
        return object.getVersion() + 1;
    }

    @Override
    protected ModerationWorkflow getWorkflow(BannerstorageCreativeWithModerationInfo moderationInfo) {
        // Ни в интерфейсе, ни в модерации пока нет поддержки других workflow для этого продукта
        // Если нужно будет добавить, то надо будет сходить в Модерацию и согласовать это
        return COMMON;
    }
}
