package ru.yandex.direct.core.entity.moderation.service.sending.bannerstorage;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.bannerstorage.client.BannerStorageClient;
import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.bannerstorage.client.model.File;
import ru.yandex.direct.bannerstorage.client.model.Parameter;
import ru.yandex.direct.bannerstorage.client.model.ParameterType;
import ru.yandex.direct.bannerstorage.client.model.Template;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.bannerstorage.client.BannerStorageClient.MAX_CREATIVES_IN_BATCH;
import static ru.yandex.direct.bannerstorage.client.BannerStorageClient.STATUS_DRAFT;
import static ru.yandex.direct.bannerstorage.client.model.CreativeInclude.IS_PREDEPLOYED;
import static ru.yandex.direct.bannerstorage.client.model.CreativeInclude.LAYOUT_CODE;
import static ru.yandex.direct.bannerstorage.client.model.CreativeInclude.PREVIEW_URL;
import static ru.yandex.direct.core.entity.moderation.repository.sending.BannerstorageCreativesSendingRepository.SUPPORTED_TEMPLATES;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class BannerstorageCreativesService {
    private static final Logger logger = LoggerFactory.getLogger(BannerstorageCreativesService.class);

    private final BannerStorageClient bannerStorageClient;
    private final CreativeRepository creativeRepository;


    public BannerstorageCreativesService(BannerStorageClient bannerStorageClient,
                                         CreativeRepository creativeRepository) {
        this.bannerStorageClient = bannerStorageClient;
        this.creativeRepository = creativeRepository;

    }

    public PreparedCreatives prepareCreatives(int shard, List<Long> ids) {
        // Получим данные по perf_creatives для этих же креативов:
        // во-первых, здесь мы возьмём только те, которые в нужном статусе
        // во-вторых, в perf_creatives лежат данные по geo
        var perfCreatives = getPerfCreatives(shard, ids);
        var perfCreativesMap = StreamEx.of(perfCreatives).toMap(t -> t.getId(), t -> t);

        // Получим их данные из bannerstorage
        var creativeIds = mapList(perfCreatives, c -> c.getId().intValue());
        var allCreatives = getCreativesFromBannerstorage(creativeIds);

        // Пропускаем креативы неизвестных шаблонов и те, у которых is_predeployed=0
        var creatives = filterCreativesForNewModeration(allCreatives);
        if (creatives.isEmpty()) {
            return new PreparedCreatives(emptyList(), emptyMap(), emptyMap(), emptyMap());
        }

        // Соберём информацию обо всех интересующих нас файловых параметрах
        var filesMap = getAllFiles(creatives);

        // Информация обо всех шаблонах (это можно закешировать, кстати)
        var templatesMap = StreamEx.of(bannerStorageClient.getTemplates()).toMap(Template::getId, t -> t);

        // Устанавливаем в bannerstorage признак того, что креативы модерируются Директом
        // Это нужно для того, чтобы креатив там не модерировался параллельно с модерацией Директа
        bannerstorageSetModeratedExternally(creatives);

        // Переводим креативы в bannerstorage в статус "Ожидает модерации"
        var creativesAwaitingModeration = bannerstorageRequestModeration(creatives);

        return new PreparedCreatives(creativesAwaitingModeration, templatesMap, filesMap, perfCreativesMap);
    }

    private List<ru.yandex.direct.core.entity.creative.model.Creative> getPerfCreatives(int shard,
                                                                                        List<Long> creativeIds) {
        var perfCreatives = creativeRepository.getCreatives(shard, creativeIds)
                .stream()
                .filter(perfCreative -> perfCreative.getStatusModerate().equals(StatusModerate.READY)
                        || perfCreative.getStatusModerate().equals(StatusModerate.SENDING))
                .collect(toList());
        logger.info(
                "Found {} ready creatives from {} events: {}",
                perfCreatives.size(), creativeIds.size(),
                perfCreatives.stream().map(c -> c.getId().toString()).collect(joining(","))
        );

        // Определим, какие из креативов привязаны к существующим баннерам
        // Для тех, которые не привязаны (баннер можно удалить пока креатив модерируется),
        // не зовём ручку в BannerStorage, чтобы статусы не расходились: отправка в Модерацию
        // не произойдёт, если к креативу не будет привязано ни одного баннера
        Set<Long> linkedWithBanners = creativeRepository.selectCreativesLinkedWithBanners(shard, creativeIds);
        var byPredicate = StreamEx.of(perfCreatives).partitioningBy(c -> linkedWithBanners.contains(c.getId()));
        var creativesToProcess = byPredicate.getOrDefault(true, emptyList());
        var creativesToSkip = byPredicate.getOrDefault(false, emptyList());
        if (!creativesToSkip.isEmpty()) {
            logger.info("Skipped {} dangling creatives (not linked to any banner): {}",
                    creativesToSkip.size(), mapList(creativesToSkip, c -> c.getId()));
        }

        // Пропустим креативы, в которых sum_geo = NULL или пустая строка
        // На лету чинить их не очень хорошая идея, т.к. модифицировать это поле при отправке в Модерацию - это
        // не то поведение, которое мы ожидаем. И мы можем что-то сделать неправильно, и только усугубим ситуацию.
        // А просто пересчитать sum_geo и отправить в Модерацию - не очень поможет, т.к. sum_geo в базе останется кривым
        // Поэтому лучше будем здесь их пропускать, устанавливая statusModerate='ERROR'
        return filterBySumGeo(shard, creativesToProcess);
    }

    private List<ru.yandex.direct.core.entity.creative.model.Creative> filterBySumGeo(
            int shard,
            List<ru.yandex.direct.core.entity.creative.model.Creative> perfCreatives
    ) {
        var byPredicate = StreamEx.of(perfCreatives)
                .partitioningBy(c -> c.getSumGeo() != null && !c.getSumGeo().isEmpty());
        var creativesToProcess = byPredicate.getOrDefault(true, emptyList());
        var creativesToSkip = byPredicate.getOrDefault(false, emptyList());
        if (!creativesToSkip.isEmpty()) {
            List<Long> skipCreativeIds = mapList(creativesToSkip, c -> c.getId());
            logger.info("Skipped with error {} creatives without sum_geo: {}",
                    creativesToSkip.size(), skipCreativeIds);

            // Установим для них статус ERROR, чтобы они не переотправлялись заново
            creativeRepository.setStatusModerate(shard, skipCreativeIds, PerfCreativesStatusmoderate.Error);
        }
        return creativesToProcess;
    }

    private List<Creative> getCreativesFromBannerstorage(List<Integer> creativeIds) {
        List<List<Integer>> batches = Lists.partition(creativeIds, MAX_CREATIVES_IN_BATCH);
        List<Creative> result = new ArrayList<>();
        for (List<Integer> batch : batches) {
            result.addAll(bannerStorageClient.getCreatives(batch, IS_PREDEPLOYED, PREVIEW_URL, LAYOUT_CODE));
        }
        return result;
    }

    private List<Creative> filterCreativesForNewModeration(List<Creative> creatives) {
        var byPredicate = StreamEx.of(creatives)
                .partitioningBy(c -> SUPPORTED_TEMPLATES.containsKey(c.getTemplateId()) && c.isPredeployed());
        var creativesToProcess = byPredicate.getOrDefault(true, emptyList());
        var creativesToSkip = byPredicate.getOrDefault(false, emptyList());
        if (!creativesToSkip.isEmpty()) {
            logger.info("Skipped {} creatives due to unsupported template_id or is_predeployed=0: {}",
                    creativesToSkip.size(), creativesToSkip.stream().map(Creative::getId).collect(toList()));
        }
        return creativesToProcess;
    }

    // Получает из bannerstorage информацию по всем файлам из указанных креативов
    private Map<Integer, File> getAllFiles(List<Creative> creatives) {
        Set<Integer> fileIds = creatives.stream()
                .map(Creative::getParameters)
                .flatMap(Collection::stream)
                .filter(p -> ParameterType.FILE.equals(p.getParamType()))
                .map(Parameter::getValues)
                .flatMap(Collection::stream)
                .filter(v -> v != null && !v.isEmpty())
                .map(Integer::valueOf)
                .collect(toSet());
        // Здесь можно сделать новую ручку в bannerstorage, которая бы отдавала эти данные массово, а не по одному
        Map<Integer, File> filesMap = new HashMap<>();
        for (var fileId : fileIds) {
            File file = bannerStorageClient.getFile(fileId);
            filesMap.put(fileId, file);
        }
        return filesMap;
    }

    private void bannerstorageSetModeratedExternally(List<Creative> creatives) {
        for (Creative creative : creatives) {
            bannerStorageClient.setModeratedExternally(creative.getId());
        }
    }

    private List<Creative> bannerstorageRequestModeration(List<Creative> creatives) {
        List<Creative> result = new ArrayList<>();
        for (Creative creative : creatives) {
            // Переводим только те, которые в статусе "Редактируется"
            // (чтобы обеспечить реентерабельность кода, если в процессе обработки что-то пошло не так)
            if (creative.getStatus() != STATUS_DRAFT) {
                logger.info("Skip requestModeration on {}: status={}", creative.getId(), creative.getStatus());
                result.add(new Creative(creative));
                continue;
            }
            // В процессе работы может возникать ситуация, когда последняя версия креатива в bannerstorage
            // больше, чем версия, которая хранится в perf_creatives (например, если редактировалась смартовая пачка
            // и возникла проблема при нотификации Директа через intapi, или если при обработке requestModeration
            // создалась новая версия. В этом случае мы хотим отправить в Модерацию ту версию, которая действительно
            // была переведена в статус "Ожидает модерации", т.к. при обработке ответа Модерации мы попросим
            // bannerstorage проставить статус на этой версии, и она должна быть именно та
            // (иначе bannerstorage ответит ошибкой наподобие "Неожиданный статус версии креатива")
            Creative afterRequestModeration = bannerStorageClient.requestModeration(creative.getId());
            result.add(
                    new Creative(creative)
                            .withVersion(afterRequestModeration.getVersion())
                            .withStatus(afterRequestModeration.getStatus())
            );
        }
        return result;
    }
}
