package ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.implementations;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.common.base.Suppliers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizationRequest;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.HrefParameterizingService;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.KeywordsInfoService;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.ParamReplacer;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.ParamReplacerFactory;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.ReplacingContext;
import ru.yandex.direct.core.entity.placements.repository.PlacementsRepository;
import ru.yandex.direct.hrefs.parameterizer.HrefParameter;

import static ru.yandex.direct.utils.CommonUtils.nvl;

@Service
public class HrefParameterizingServiceImpl implements HrefParameterizingService {
    private Map<String, ParamReplacer> replacersMap;
    private Pattern pattern;
    private final KeywordsInfoService keywordsInfoService;
    private final PlacementsRepository placementsRepository;

    @Autowired
    public HrefParameterizingServiceImpl(KeywordsInfoService keywordsInfoService,
                                         PlacementsRepository placementsRepository) {
        this.keywordsInfoService = keywordsInfoService;
        this.placementsRepository = placementsRepository;
        pattern = Pattern.compile("(\\{[^}]+})");
        buildReplacersMap();
    }

    private void buildReplacersMap() {
        ParamReplacerFactory paramReplacerFactory = new ParamReplacerFactory();

        replacersMap = new HashMap<>();

        Supplier<List<String>> domainsSupplier = Suppliers.memoizeWithExpiration(
                () -> placementsRepository.getRandomPlacementsDomains(50), 1, TimeUnit.HOURS);

        for (HrefParameter parameter : HrefParameter.values()) {
            ParamReplacer replacer = paramReplacerFactory.get(parameter, domainsSupplier);

            for (String token : parameter.getKeys()) {
                replacersMap.put(token, replacer);
            }
        }
    }

    @Override
    public List<String> parameterize(int shard, List<HrefParameterizationRequest> requests) {
        List<String> result = new ArrayList<>();

        List<Long> pids = requests.stream()
                .map(h -> h.getParams().getPid())
                .distinct()
                .collect(Collectors.toList());

        ReplacingContext context = null;

        for (HrefParameterizationRequest request : requests) {
            Matcher matcher = pattern.matcher(request.getHref());

            if (matcher.groupCount() > 0 && context == null) {
                Map<Long, KeywordsInfoService.KeywordInfo> keywordInfoMap = keywordsInfoService.loadKeywordInfo(pids,
                        shard);

                context = new ReplacingContextImpl(keywordInfoMap);
            }

            ReplacingContext finalContext = context;
            result.add(matcher.replaceAll(f -> matchResultToReplacement(f, finalContext, request)));
        }

        return result;
    }

    private String matchResultToReplacement(MatchResult f, ReplacingContext context,
                                            HrefParameterizationRequest request) {
        String token = request.getHref().substring(f.start() + 1, f.end() - 1);

        if (replacersMap.containsKey(token)) {
            String replacement = replacersMap.get(token).apply(request.getParams(), context);
            replacement = nvl(replacement, "");
            return URLEncoder.encode(replacement, StandardCharsets.UTF_8);
        } else {
            return "";
        }
    }
}
