package ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.implementations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.advq.query.ast.Expression;
import ru.yandex.advq.query.ast.ExpressionTransform;
import ru.yandex.advq.query.ast.Intersection;
import ru.yandex.advq.query.ast.QuotedWords;
import ru.yandex.advq.query.ast.SquareBrackets;
import ru.yandex.advq.query.ast.Word;
import ru.yandex.direct.core.entity.moderation.repository.sending.KeywordsInfoRepository;
import ru.yandex.direct.core.entity.moderation.service.sending.hrefs.parameterizer.KeywordsInfoService;
import ru.yandex.direct.libs.keywordutils.parser.KeywordParser;
import ru.yandex.direct.libs.keywordutils.parser.ReplaceCharsExpressionTransform;

import static ru.yandex.advq.query.ast.WordKind.RAW;
import static ru.yandex.direct.libs.keywordutils.parser.KeywordWithMinusesBuilder.buildKeywordWithMinusesFrom;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Service
public class KeywordsInfoServiceImpl implements KeywordsInfoService {
    private final KeywordsInfoRepository keywordsInfoRepository;
    private static final Logger logger = LoggerFactory.getLogger(KeywordsInfoServiceImpl.class);


    @Autowired
    public KeywordsInfoServiceImpl(KeywordsInfoRepository keywordsInfoRepository) {
        this.keywordsInfoRepository = keywordsInfoRepository;
    }

    private String prepareKeyword(String originalKeyword) {
        try {
            Expression expression = KeywordParser.buildExpression(originalKeyword)
                    .accept(ReplaceKindTransform.INSTANCE)
                    .accept(ReplaceCharsExpressionTransform.INSTANCE);

            return buildKeywordWithMinusesFrom(expression).getKeyword().toString();
        } catch (Exception e) {
            logger.error("Can't prepare keyword " + originalKeyword, e);
            return originalKeyword;
        }
    }

    @Override
    public Map<Long, KeywordInfo> loadKeywordInfo(List<Long> pids, int shard) {
        return loadFromDb(pids, shard);
    }

    private static class ReplaceKindTransform extends ExpressionTransform {

        public static final ReplaceKindTransform INSTANCE = new ReplaceKindTransform();

        @Override
        public Expression visitWord(Word word) {
            return new Word(RAW, word.getText());
        }

        @Override
        public Expression visitSquareBrackets(SquareBrackets expr) {
            List<Expression> transformed = new ArrayList<>();

            for (Word original : expr.getWords()) {
                transformed.add(original.accept(this));
            }

            return new Intersection(transformed);
        }

        @Override
        public Expression visitQuotedWords(QuotedWords expr) {
            Expression original = expr.getChild();
            return original.accept(this);
        }
    }

    private Map<Long, KeywordInfo> loadFromDb(List<Long> pids, int shard) {

        Map<Long, KeywordInfo> fromDb = keywordsInfoRepository.loadFromDb(shard, pids);
        Map<Long, KeywordInfo> result = new HashMap<>();

        for (var row : fromDb.entrySet()) {

            KeywordInfo rawInfo = row.getValue();

            KeywordInfo keywordInfo = new KeywordInfo(
                    ifNotNull(rawInfo.getKeyword(), this::prepareKeyword),
                    rawInfo.getParam1(),
                    rawInfo.getParam2(),
                    rawInfo.getPhraseId(),
                    rawInfo.getRetargetingId());

            result.put(row.getKey(), keywordInfo);
        }

        return result;
    }

}
