package ru.yandex.direct.core.entity.moderationdiag.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.dbschema.ppcdict.enums.ModerateDiagsAllowFirstAid;
import ru.yandex.direct.dbschema.ppcdict.enums.ModerateDiagsBadReason;
import ru.yandex.direct.dbschema.ppcdict.enums.ModerateDiagsUnbanIsProhibited;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppcdict.enums.ModerateDiagsUnbanIsProhibited.No;
import static ru.yandex.direct.dbschema.ppcdict.enums.ModerateDiagsUnbanIsProhibited.Yes;
import static ru.yandex.direct.dbschema.ppcdict.tables.ModerateDiags.MODERATE_DIAGS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Repository
@ParametersAreNonnullByDefault
public class ModerationDiagRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<ModerationDiag> moderationDiagMapper;

    @Autowired
    public ModerationDiagRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.moderationDiagMapper = createMapper();
    }

    public List<ModerationDiag> fetch(Collection<Long> moderationDiagIds) {
        return fetch(null, moderationDiagIds);
    }

    public List<ModerationDiag> fetch(ModerationDiagType type) {
        return fetch(type, null);
    }

    public List<ModerationDiag> fetch(@Nullable ModerationDiagType type, @Nullable Collection<Long> moderationDiagIds) {
        checkArgument(type != null || moderationDiagIds != null,
                "ModerationDiagType or ModerationDiagIds must be specified");

        var selectJoinStep = dslContextProvider.ppcdict()
                .select(moderationDiagMapper.getFieldsToRead())
                .from(MODERATE_DIAGS);

        List<Condition> conditions = new ArrayList<>();

        if (type != null) {
            conditions.add(MODERATE_DIAGS.TYPE.eq(ModerationDiagType.toSource(type)));
        }
        if (moderationDiagIds != null) {
            conditions.add(MODERATE_DIAGS.DIAG_ID.in(moderationDiagIds));
        }

        List<ModerationDiag> result = selectJoinStep.where(conditions)
                .fetch(moderationDiagMapper::fromDb);
        result.forEach(diag -> diag.setDiagText(nvl(diag.getFullText(), diag.getShortText())));
        return result;
    }

    public void insertModerationDiagOrUpdateTexts(Collection<ModerationDiag> moderationDiags) {
        new InsertHelper<>(dslContextProvider.ppcdict(), MODERATE_DIAGS)
                .addAll(moderationDiagMapper, moderationDiags)
                .onDuplicateKeyUpdate()
                .set(MODERATE_DIAGS.TEXT_FULL, MySQLDSL.values(MODERATE_DIAGS.TEXT_FULL))
                .set(MODERATE_DIAGS.TEXT, MySQLDSL.values(MODERATE_DIAGS.TEXT))
                .execute();
    }

    public void updateUnbanIsProhibited(Collection<Long> moderationDiagIds, boolean unbanIsProhibited) {
        if (moderationDiagIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppcdict()
                .update(MODERATE_DIAGS)
                .set(MODERATE_DIAGS.UNBAN_IS_PROHIBITED, unbanIsProhibited ? Yes : No)
                .where(MODERATE_DIAGS.DIAG_ID.in(moderationDiagIds))
                .execute();
    }

    @Nonnull
    private JooqMapperWithSupplier<ModerationDiag> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(ModerationDiag::new)
                .map(property(ModerationDiag.ID, MODERATE_DIAGS.DIAG_ID))
                .map(convertibleProperty(ModerationDiag.TYPE, MODERATE_DIAGS.TYPE,
                        ModerationDiagType::fromSource,
                        ModerationDiagType::toSource))
                .map(property(ModerationDiag.SHORT_TEXT, MODERATE_DIAGS.TEXT))
                .map(property(ModerationDiag.FULL_TEXT, MODERATE_DIAGS.TEXT_FULL))
                .map(booleanProperty(ModerationDiag.ALLOW_FIRST_AID, MODERATE_DIAGS.ALLOW_FIRST_AID,
                        ModerateDiagsAllowFirstAid.class))
                .map(booleanProperty(ModerationDiag.STRONG_REASON, MODERATE_DIAGS.BAD_REASON,
                        ModerateDiagsBadReason.class))
                .map(booleanProperty(ModerationDiag.UNBAN_IS_PROHIBITED, MODERATE_DIAGS.UNBAN_IS_PROHIBITED,
                        ModerateDiagsUnbanIsProhibited.class))
                .map(property(ModerationDiag.TOKEN, MODERATE_DIAGS.TOKEN))
                .build();
    }
}
