package ru.yandex.direct.core.entity.moderationdiag.service;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.moderationdiag.model.DiagReasonTitleAndDescription;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.moderationdiag.repository.ModerationDiagRepository;

import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.StringUtils.htmlToPlainText;

@Service
@ParametersAreNonnullByDefault
public class ModerationDiagService {
    private static final Logger logger = LoggerFactory.getLogger(ModerationDiagService.class);
    //https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/ModerateDiagnosis.pm?rev=r8057070#L45
    private static final Set<Long> DETAILS_LINK = Set.of(
            102L, //Авторское право
            2L, // Не соответствует действующему законодательству
            99L, // Сайт не соответствует требованиям
            1L, // Рекламная политика Яндекса не соблюдена
            69L, // Политическая реклама
            43L // Требуется лицензия, сертификат или ГП
    );

    private static final int RELOAD_INTERVAL_IN_MINUTES = 10;

    private final ModerationDiagRepository moderationDiagRepository;
    private final Cache<ModerationDiagType, Map<Long, ModerationDiag>> moderateDiagCache;

    @Autowired
    public ModerationDiagService(ModerationDiagRepository moderationDiagRepository) {
        this.moderationDiagRepository = moderationDiagRepository;
        moderateDiagCache = CacheBuilder.newBuilder()
                .expireAfterWrite(RELOAD_INTERVAL_IN_MINUTES, TimeUnit.MINUTES)
                .build();
    }

    public Map<Long, ModerationDiag> get(ModerationDiagType type) {
        try {
            return moderateDiagCache.get(type, () -> createDiagMap(type));
        } catch (ExecutionException e) {
            throw new RuntimeException("Error during getting ModerateDiagMap", e.getCause());
        }
    }

    //for test purposes
    public void invalidateAll() {
        moderateDiagCache.invalidateAll();
    }

    private Map<Long, ModerationDiag> createDiagMap(ModerationDiagType type) {
        return moderationDiagRepository.fetch(type).stream()
                .collect(Collectors.toMap(ModerationDiag::getId, Function.identity()));
    }

    @Nullable
    public static DiagReasonTitleAndDescription extractTitleAndDescFromReasonHtml(String reasonAsHtml) {
        try {
            JsonNode json = fromJson(reasonAsHtml);
            String content = htmlToPlainText(json.get("content").asText());
            String title = htmlToPlainText(json.get("selector").get(".title").asText());

            return DiagReasonTitleAndDescription.builder()
                    .setTitle(title)
                    .setDescription(content)
                    .build();
        } catch (RuntimeException e) {
            logger.error("cannot parse reason", e);
            return null;
        }
    }

    public static boolean needShowDetailsUrl(ModerationDiag moderationDiag) {
        return DETAILS_LINK.contains(moderationDiag.getId());
    }
}
