package ru.yandex.direct.core.entity.moderationreason;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Functions;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;

/**
 * Помощник для построения объекта {@code Map<ModerationReasonObjectType, Map<Long, Set<Long>>>},
 * являющегося отображением из типа модерируемого объекта в отображение id подобъектов данных типов
 * в id их баннеров-владельцев. Позволяет удобно отбирать id подобъектов соответствующих типов.
 *
 * @param <T> тип баннера-владельца
 */
@ParametersAreNonnullByDefault
public class ModerationReasonFilterBuilder<T extends Banner> {
    private final Class<T> clazz;
    private final Predicate<T> predicate;
    private final Function<T, Collection<Long>> getIds;
    private final Collection<ModerationReasonObjectType> objectTypes;

    public ModerationReasonFilterBuilder(Class<T> clazz, Predicate<T> predicate, Function<T, Collection<Long>> getIds,
                                         Collection<ModerationReasonObjectType> objectTypes) {
        this.clazz = clazz;
        this.predicate = predicate;
        this.getIds = getIds;
        this.objectTypes = objectTypes;
    }

    public Class<T> getBannerClass() {
        return clazz;
    }

    public void process(Collection<Banner> banners, Map<ModerationReasonObjectType, Map<Long, Set<Long>>> filter) {
        Map<Long, Set<Long>> ids =
                StreamEx.of(banners)
                        .select(clazz)
                        .filter(predicate)
                        .mapToEntry(getIds, Banner::getId)
                        .nonNullKeys()
                        .flatMapKeys(StreamEx::of)
                        .nonNullKeys()
                        .groupingTo(HashSet::new);
        if (!ids.isEmpty()) {
            for (var objectType : objectTypes) {
                filter.merge(objectType, ids, ModerationReasonFilterBuilder::mergeMaps);
            }
        }
    }

    public static <K, V, C extends Collection<V>> Map<K, C> mergeMaps(Map<K, C> m1, Map<K, C> m2) {
        for (var entry : m2.entrySet()) {
            m1.merge(entry.getKey(), entry.getValue(), (s1, s2) -> {
                s1.addAll(s2);
                return s1;
            });
        }
        return m1;
    }

    public static <T extends Banner> ModerationReasonFilterBuilder<T> simple(Class<T> clazz,
                                                                             Collection<ModerationReasonObjectType> objectTypes) {
        return new ModerationReasonFilterBuilder<>(clazz, banner -> true, Functions.compose(List::of, Banner::getId),
                objectTypes);
    }

    public static <T extends Banner> ModerationReasonFilterBuilder<T> withPredicate(Class<T> clazz,
                                                                                    Function<T, Object> getField,
                                                                                    Collection<ModerationReasonObjectType> objectTypes) {
        return new ModerationReasonFilterBuilder<>(clazz, banner -> getField.apply(banner) != null,
                Functions.compose(List::of, Banner::getId), objectTypes);
    }

    public static <T extends Banner> ModerationReasonFilterBuilder<T> withMapper(Class<T> clazz,
                                                                                 Function<T, Long> getId,
                                                                                 Collection<ModerationReasonObjectType> objectTypes) {
        return withMultiMapper(clazz,
                banner -> Optional.ofNullable(getId.apply(banner)).map(List::of).orElse(null), objectTypes);
    }

    public static <T extends Banner> ModerationReasonFilterBuilder<T> withMultiMapper(Class<T> clazz,
                                                                                      Function<T, Collection<Long>> getIds,
                                                                                      Collection<ModerationReasonObjectType> objectTypes) {
        return new ModerationReasonFilterBuilder<>(clazz, banner -> true, getIds, objectTypes);
    }
}
