package ru.yandex.direct.core.entity.moderationreason.repository

import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.annotation.JsonProperty
import com.fasterxml.jackson.core.type.TypeReference
import one.util.streamex.EntryStream
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReason
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonDetailed
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.utils.YamlUtils


/**
 * В базе причины хранятся в поле `reason` в следующем формате:
 * ```
 * ---
 * -
 *   id: 157
 *   comment: ''
 *   screenshots:
 *     - "https://storage.mds.yandex.net/get-mturk/57024/78ad7e07-2e13-4205-9726-70e2d26022de"
 *   list:
 *     -
 *       id: '20849359827'
 * -
 *   id: 287
 * -
 *   id: 293
 *   list:
 *     -
 *       id: '20849359827'
 * ```
 *
 * Здесь `.[0].list[0].id` &mdash; id под-объекта, к которому привязана причина отклонения.
 * Например, id фразы (сама причина хранится на группе)
 */
object ModerationReasonMapping {

    private val logger = LoggerFactory.getLogger(ModerationReasonMapping::class.java)

    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonIgnoreProperties(ignoreUnknown = true)
    private class ReasonInfo(
        @JsonProperty("id")
        val id: Long,
        @JsonProperty("comment")
        val comment: String?,
        @JsonProperty("screenshots")
        val screenshots: List<String>?,
        @JsonProperty("list")
        val itemIds: List<ReasonSubObjectInfo>?,
    )

    @JsonIgnoreProperties(ignoreUnknown = true)
    private class ReasonSubObjectInfo(
        @JsonProperty("id")
        val id: Long,
    )

    @JvmStatic
    fun reasonsFromDbFormat(reasons: String?): List<ModerationReasonDetailed> {
        val reasonList: List<ReasonInfo> = parseReasons(reasons)
            ?: return emptyList()

        return reasonList.map { reason ->
            ModerationReasonDetailed()
                .withId(reason.id)
                .withComment(reason.comment)
                .withScreenshots(reason.screenshots)
                .withItemIds(reason.itemIds?.map { it.id })
        }
    }

    @JvmStatic
    fun reasonsToDbFormat(reasons: List<ModerationReasonDetailed?>?): String {
        if (reasons.isNullOrEmpty()) {
            return ""
        }

        val reasonsList = reasons.filterNotNull().map { reason ->
            if (reason.itemIds != null && reason.itemIds.filter { it == null }.count() > 0) {
                logger.error("Null ids in reason $reason")
            }
            ReasonInfo(
                id = reason.id,
                comment = reason.comment,
                screenshots = reason.screenshots,
                itemIds = reason.itemIds?.filterNotNull()?.map { id -> ReasonSubObjectInfo(id) },
            )
        }

        return YamlUtils.toYamlWithoutSplitLines(reasonsList)
    }

    @JvmStatic
    fun getSubObjectIds(moderationReason: ModerationReason): Set<Long> {
        return moderationReason.reasons.orEmpty()
            .mapNotNull { reason -> reason.itemIds }
            .flatten().toSet()
    }

    @JvmStatic
    fun assetsReasonsFromDbFormat(
            assetsReasons: String?
    ): Map<BannerAssetType, Set<Long>>? =
            if (assetsReasons.isNullOrBlank()) null
            else JsonUtils.fromJson(assetsReasons, object: TypeReference<Map<String, Set<Long>>>() {})
                    .mapKeys{ BannerAssetType.fromTypedValue(it.key) }

    @JvmStatic
    fun assetsReasonsToDbFormat(
            assetsReasons: Map<BannerAssetType, Set<Long>>?
    ): String? = if (assetsReasons != null) JsonUtils.toJson(assetsReasons) else null

    private fun parseReasons(reasons: String?): List<ReasonInfo>? {
        if (reasons.isNullOrBlank()) {
            return null
        }

        return try {
            YamlUtils.fromYaml(reasons, object : TypeReference<List<ReasonInfo>>() {})
        } catch (e: IllegalArgumentException) {
            logger.warn("Can't parse reasons", e)
            null
        }
    }
}
