package ru.yandex.direct.core.entity.moderationreason.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.moderationreason.model.ModerationReason;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusModerate;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusPostModerate;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonStatusSending;
import ru.yandex.direct.dbschema.ppc.enums.ModReasonsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.ModReasonsStatussending;
import ru.yandex.direct.dbschema.ppc.enums.ModReasonsType;
import ru.yandex.direct.dbschema.ppc.tables.records.ModReasonsRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.function.Function.identity;
import static ru.yandex.direct.dbschema.ppc.tables.ModReasons.MOD_REASONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class ModerationReasonRepository {
    private static final int CHUNK_SIZE = 5000;
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<ModerationReason> moderationReasonMapper;

    @Autowired
    public ModerationReasonRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        moderationReasonMapper = createMapper();
    }

    /**
     * Формирует условия для выборки из базы с не более чем {@link #CHUNK_SIZE} объектов в каждом условии.
     */
    private static List<Condition> getConditions(Map<ModerationReasonObjectType, List<Long>> moderationReasonsMap) {
        AtomicLong counter = new AtomicLong();
        return EntryStream.of(moderationReasonsMap)
                .flatMapValues(StreamEx::of)
                .map(identity())
                .groupRuns((e1, e2) -> counter.incrementAndGet() % CHUNK_SIZE != 0)
                .map(group -> EntryStream.of(group.iterator())
                        .collapseKeys()
                        .mapKeys(ModerationReasonObjectType::toSource)
                        .reduce(DSL.falseCondition(), createCondition(), Condition::or))
                .toList();
    }

    @Nonnull
    private static BiFunction<Condition, Map.Entry<ModReasonsType, List<Long>>, Condition> createCondition() {
        return (condition, entry) -> condition.or(
                MOD_REASONS.TYPE.eq(entry.getKey()).and(MOD_REASONS.ID.in(entry.getValue()))
        );
    }

    public List<ModerationReason> fetchRejected(int shard, ModerationReasonObjectType objectType,
                                                Collection<Long> objectIds) {
        return fetchRejected(dslContextProvider.ppc(shard).configuration(), objectType, objectIds);
    }

    public List<ModerationReason> fetchRejected(Configuration configuration, ModerationReasonObjectType objectType,
                                                Collection<Long> objectIds) {
        return DSL.using(configuration)
                .select(moderationReasonMapper.getFieldsToRead())
                .from(MOD_REASONS)
                .where(MOD_REASONS.TYPE.eq(ModerationReasonObjectType.toSource(objectType)))
                .and(MOD_REASONS.ID.in(objectIds))
                .and(MOD_REASONS.STATUS_MODERATE.eq(ModReasonsStatusmoderate.No))
                .fetch(moderationReasonMapper::fromDb);
    }

    public List<ModerationReason> fetchRejected(int shard,
                                                Map<ModerationReasonObjectType, List<Long>> objectIdsByModReasonType) {
        if (objectIdsByModReasonType.isEmpty()) {
            return Collections.emptyList();
        }
        List<Condition> conditions = getConditions(objectIdsByModReasonType);
        List<ModerationReason> reasons = new ArrayList<>();
        for (Condition condition : conditions) {
            SelectConditionStep<Record> fetcher = getFetcher(shard);
            reasons.addAll(fetcher.and(condition).fetch(moderationReasonMapper::fromDb));
        }
        return reasons;
    }

    /**
     * Удаляет записи из mod_reasons в шарде по паре (ClientID, cid) в статусе statusSending No и Sending
     *
     * @param shard    шард
     * @param clientId id клиента
     * @param cid      id кампании
     * @return количество удалённых строк
     */
    public int deleteUnsentModerationReasons(int shard, ClientId clientId, long cid) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(MOD_REASONS)
                .where(MOD_REASONS.CLIENT_ID.eq(clientId.asLong())
                        .and(MOD_REASONS.CID.eq(cid))
                        .and(MOD_REASONS.STATUS_SENDING
                                .in(ModReasonsStatussending.Sending, ModReasonsStatussending.No)))
                .execute();
    }

    /**
     * Возвращает список rid из mod_reasons в шарде старше указанного времени в статусе statusModerate Yes
     * В базе может накопиться слишком много данных, явно количество записей лимитом
     *
     * @param shard          шард
     * @param borderDateTime максимальный возраст принятой записи
     * @param limit          максимальное количество записей, которые получаем из базы
     * @return список rid
     */
    @QueryWithoutIndex("Получаем записи для удаления, выполняется в jobs")
    public List<Long> getAcceptedModerationReasonsIdsOlderThanDateTime(int shard, LocalDateTime borderDateTime, Integer limit) {
        return dslContextProvider.ppc(shard)
                .select(MOD_REASONS.RID)
                .from(MOD_REASONS)
                .where(MOD_REASONS.TIME_CREATED.lessThan(borderDateTime))
                .and(MOD_REASONS.STATUS_MODERATE.eq(ModReasonsStatusmoderate.Yes))
                .limit(limit)
                .fetch(MOD_REASONS.RID);
    }

    /**
     * Удаляет записи из mod_reasons в шарде по списку rid старше указанного времени в статусе statusModerate Yes
     *
     * @param shard          шард
     * @param ids            список rid
     * @param borderDateTime максимальный возраст принятой записи
     * @return количество удалённых строк
     */
    public int deleteAcceptedModerationReasonsOlderThanDateTime(int shard, Collection<Long> ids,
                                                                LocalDateTime borderDateTime) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(MOD_REASONS)
                .where(MOD_REASONS.RID.in(ids))
                .and(MOD_REASONS.TIME_CREATED.lessThan(borderDateTime))
                .and(MOD_REASONS.STATUS_MODERATE.eq(ModReasonsStatusmoderate.Yes))
                .execute();
    }

    /**
     * Добавляет записи в mod_reasons.
     * Если запись уже существует, то у неё обновится statusSending, statusModerate, statusPostModerate, timeCreated,
     * reason.
     */
    public void insertOrUpdateModerationReasons(Configuration jooqConfig,
                                                Collection<ModerationReason> moderationReasons) {
        InsertHelper<ModReasonsRecord> insertHelper = new InsertHelper<>(jooqConfig.dsl(), MOD_REASONS);
        insertHelper.addAll(moderationReasonMapper, moderationReasons);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(MOD_REASONS.STATUS_SENDING, MySQLDSL.values(MOD_REASONS.STATUS_SENDING))
                    .set(MOD_REASONS.STATUS_MODERATE, MySQLDSL.values(MOD_REASONS.STATUS_MODERATE))
                    .set(MOD_REASONS.STATUS_POST_MODERATE, MySQLDSL.values(MOD_REASONS.STATUS_POST_MODERATE))
                    .set(MOD_REASONS.TIME_CREATED, MySQLDSL.values(MOD_REASONS.TIME_CREATED))
                    .set(MOD_REASONS.REASON, MySQLDSL.values(MOD_REASONS.REASON));
        }

        insertHelper.executeIfRecordsAdded();
    }

    public void deleteFromModReasons(int shard, Collection<Long> ids, ModReasonsType type) {
        deleteFromModReasons(dslContextProvider.ppc(shard), ids, type);
    }

    public void deleteFromModReasons(Configuration configuration, Collection<Long> ids, ModReasonsType type) {
        deleteFromModReasons(DSL.using(configuration), ids, type);
    }

    public void deleteFromModReasons(DSLContext dslContext, Collection<Long> ids, ModReasonsType type) {
        if (ids.isEmpty()) {
            return;
        }
        dslContext
                .deleteFrom(MOD_REASONS)
                .where(MOD_REASONS.ID.in(ids))
                .and(MOD_REASONS.TYPE.eq(type))
                .execute();
    }

    private SelectConditionStep<Record> getFetcher(int shard) {
        return dslContextProvider.ppc(shard)
                .select(moderationReasonMapper.getFieldsToRead())
                .from(MOD_REASONS)
                .where(MOD_REASONS.STATUS_MODERATE.eq(ModReasonsStatusmoderate.No));
    }

    @Nonnull
    private JooqMapperWithSupplier<ModerationReason> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(ModerationReason::new)
                .map(property(ModerationReason.RID, MOD_REASONS.RID))
                .map(property(ModerationReason.OBJECT_ID, MOD_REASONS.ID))
                .map(convertibleProperty(ModerationReason.OBJECT_TYPE, MOD_REASONS.TYPE,
                        ModerationReasonObjectType::fromSource,
                        ModerationReasonObjectType::toSource))
                .map(property(ModerationReason.CLIENT_ID, MOD_REASONS.CLIENT_ID))
                .map(property(ModerationReason.CAMPAIGN_ID, MOD_REASONS.CID))
                .map(convertibleProperty(ModerationReason.STATUS_SENDING, MOD_REASONS.STATUS_SENDING,
                        ModerationReasonStatusSending::fromSource,
                        ModerationReasonStatusSending::toSource))
                .map(convertibleProperty(ModerationReason.STATUS_MODERATE, MOD_REASONS.STATUS_MODERATE,
                        ModerationReasonStatusModerate::fromSource,
                        ModerationReasonStatusModerate::toSource))
                .map(convertibleProperty(ModerationReason.STATUS_POST_MODERATE, MOD_REASONS.STATUS_POST_MODERATE,
                        ModerationReasonStatusPostModerate::fromSource,
                        ModerationReasonStatusPostModerate::toSource))
                .map(property(ModerationReason.CREATED_AT, MOD_REASONS.TIME_CREATED))
                .map(convertibleProperty(ModerationReason.REASONS, MOD_REASONS.REASON,
                        ModerationReasonMapping::reasonsFromDbFormat,
                        ModerationReasonMapping::reasonsToDbFormat))
                .map(convertibleProperty(ModerationReason.ASSETS_REASONS, MOD_REASONS.ASSETS_REASONS,
                        ModerationReasonMapping::assetsReasonsFromDbFormat,
                        ModerationReasonMapping::assetsReasonsToDbFormat))
                .build();
    }
}
