package ru.yandex.direct.core.entity.moderationreason.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReason;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonMapping;
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class ModerationKeywordService {
    private final KeywordRepository keywordRepository;
    private final ShardHelper shardHelper;
    private final ModerationReasonRepository moderationReasonRepository;

    @Autowired
    public ModerationKeywordService(
            KeywordRepository keywordRepository, ShardHelper shardHelper,
            ModerationReasonRepository moderationReasonRepository) {
        this.keywordRepository = keywordRepository;
        this.shardHelper = shardHelper;
        this.moderationReasonRepository = moderationReasonRepository;
    }

    @QueryWithForbiddenShardMapping("pid")
    public Map<Long, List<Keyword>> getKeywordsByGroupIds(List<Long> groupIds) {
        Map<Integer, List<Long>> shards =
                shardHelper.groupByShard(groupIds, ShardKey.PID).getShardedDataMap();

        Map<Long, List<Keyword>> result = new HashMap<>();
        for (Map.Entry<Integer, List<Long>> entry : shards.entrySet()) {
            List<ModerationReason> reasons = moderationReasonRepository.fetchRejected(entry.getKey(),
                    ModerationReasonObjectType.PHRASES, entry.getValue());

            List<Long> keywordIds = StreamEx.of(reasons)
                    .flatCollection(ModerationReasonMapping::getSubObjectIds)
                    .toList();
            List<Keyword> keywords = keywordRepository.getKeywordsByIds(entry.getKey(), keywordIds);
            for (ModerationReason reason : reasons) {
                Set<Long> listReasons = listToSet(ModerationReasonMapping.getSubObjectIds(reason));
                List<Keyword> filteredWords = StreamEx.of(keywords)
                        .filter(k -> listReasons.contains(k.getId())).toList();
                result.computeIfAbsent(reason.getObjectId(), k -> new ArrayList<>())
                        .addAll(filteredWords);
            }
        }
        return result;
    }
}
