package ru.yandex.direct.core.entity.moderationreason.service;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReportResponse;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.moderationdiag.ModerationDiagDataConverterKt.convertList;
import static ru.yandex.direct.core.entity.moderationreason.service.ModerationDiagConverter.badPhrasesConverter;
import static ru.yandex.direct.core.entity.moderationreason.service.ModerationDiagConverter.convertMap;

@Service
public class ModerationReasonAggregatorService implements ModerationReasonAggregator {
    private final ModerationReasonService moderationReasonService;
    private final ModerationKeywordService moderationKeywordService;
    private final PermissionService permissionService;
    private final AgencyInfoService agencyInfoService;
    private final BannerTypedRepository bannerTypedRepository;
    private final ShardHelper shardHelper;

    private static final Logger logger = LoggerFactory.getLogger(ModerationReasonAggregatorService.class);

    @Autowired
    public ModerationReasonAggregatorService(
            ModerationReasonService moderationReasonService,
            ModerationKeywordService moderationKeywordService,
            PermissionService permissionService,
            AgencyInfoService agencyInfoService,
            BannerTypedRepository bannerTypedRepository,
            ShardHelper shardHelper) {
        this.moderationReasonService = moderationReasonService;
        this.moderationKeywordService = moderationKeywordService;
        this.permissionService = permissionService;
        this.agencyInfoService = agencyInfoService;
        this.bannerTypedRepository = bannerTypedRepository;
        this.shardHelper = shardHelper;
    }

    public Optional<ModerationReportResponse> showDiags(
            long operatorUid, ClientId clientId,
            List<Long> creativeRejectionReasonIds, List<Long> additionsItemIds, Long bannerId
    ) {
        ModerationReportResponse moderationReportResponse = new ModerationReportResponse();

        if (bannerId != null) {
            final var shard = shardHelper.getShardByClientId(clientId);
            final var banners = bannerTypedRepository.getTyped(shard, singletonList(bannerId));

            if (CollectionUtils.isEmpty(banners)) {
                logger.warn("No such banner " + bannerId);
                return Optional.empty();
            }

            BannerWithSystemFields banner = (BannerWithSystemFields) banners.get(0);
            if (!permissionService.operatorHasPermission(operatorUid, clientId, banner)) {
                return Optional.empty();
            }

            if (additionsItemIds != null) {
                Map<Long, List<ModerationDiag>> calloutReasons = moderationReasonService.getCalloutRejectedReasons(
                        bannerId, additionsItemIds);

                moderationReportResponse.setCalloutDiags(convertMap(calloutReasons));
            } else {
                agencyInfoService.fillBannerAgencyInfo(clientId, moderationReportResponse, banner.getCampaignId());
                fillAllBannerRejectedReasons(moderationReportResponse, banner,
                        moderationKeywordService.getKeywordsByGroupIds(singletonList(banner.getAdGroupId())),
                        getModerationReasonObjectTypeListMap(banner,
                                moderationReasonService.getReasons(singletonList(banner.getId()))));
            }
        } else {
            if (!permissionService.operatorHasPermission(operatorUid, clientId)) {
                return Optional.empty();
            }

            if (creativeRejectionReasonIds != null) {
                List<ModerationDiag> moderationDiags = moderationReasonService.getModerationDiagsByIds(
                        creativeRejectionReasonIds);

                moderationReportResponse.setCreativesDiags(convertList(moderationDiags));
            }
        }

        return Optional.of(moderationReportResponse);
    }

    static void fillAllBannerRejectedReasons(ModerationReportResponse moderationReportResponse,
                                             BannerWithSystemFields banner,
                                             Map<Long, List<Keyword>> keywordsByGroupIds,
                                             Map<ModerationReasonObjectType, List<ModerationDiag>> bannerReasons) {
        moderationReportResponse.setBid(String.valueOf(banner.getId()));
        if (banner instanceof BannerWithVcard) {
            BannerWithVcard bannerWithVcard = (BannerWithVcard) banner;
            moderationReportResponse.setVcardId(String.valueOf(bannerWithVcard.getVcardId()));
        }
        if (bannerReasons != null) {
            Map<String, List<ModerationDiagData>> bannerDiags = new LinkedHashMap<>();
            bannerReasons.forEach((k, v) -> bannerDiags.put(k.toString().toLowerCase(), convertList(v)));
            moderationReportResponse.setBannerDiags(bannerDiags);
        }
        fillBadPhrases(moderationReportResponse, banner, keywordsByGroupIds);
        if (bannerReasons != null) {
            fillBadImage(moderationReportResponse, bannerReasons, banner);
            fillOtherBad(moderationReportResponse, bannerReasons);
        }
    }

    private static Map<ModerationReasonObjectType, List<ModerationDiag>> getModerationReasonObjectTypeListMap(
            Banner banner, Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> bannerReasonsMap) {
        Map<ModerationReasonObjectType, List<ModerationDiag>> bannerReasons = null;
        if (!bannerReasonsMap.isEmpty()) {
            bannerReasons = bannerReasonsMap.get(banner.getId());
        }
        return bannerReasons;
    }

    private static void fillOtherBad(ModerationReportResponse moderationReport,
                                     Map<ModerationReasonObjectType, List<ModerationDiag>> bannerReasons) {
        moderationReport.setBadSitelinks(fillOneBadEntry(bannerReasons, ModerationReasonObjectType.SITELINKS_SET));
        moderationReport.setBadDisplayHref(fillOneBadEntry(bannerReasons, ModerationReasonObjectType.DISPLAY_HREF));
        moderationReport.setBadTurbolandings(fillOneBadEntry(bannerReasons, ModerationReasonObjectType.TURBOLANDING));
    }

    private static Map<String, List<ModerationDiagData>> fillOneBadEntry(
            Map<ModerationReasonObjectType, List<ModerationDiag>> bannerReasons,
            ModerationReasonObjectType moderationReasonObjectType) {
        Map<String, List<ModerationDiagData>> reason = null;
        List<ModerationDiag> moderationDiags = bannerReasons.get(moderationReasonObjectType);
        if (moderationDiags != null && !moderationDiags.isEmpty()) {
            reason = ImmutableMap.of("reason", convertList(moderationDiags));
        }
        return reason;
    }

    private static void fillBadImage(
            ModerationReportResponse moderationReportResponse,
            Map<ModerationReasonObjectType, List<ModerationDiag>> bannerReasons, Banner banner
    ) {
        if (banner instanceof BannerWithBannerImage) {
            BannerWithBannerImage bannerWithBannerImage = (BannerWithBannerImage) banner;
            if (bannerReasons != null && bannerWithBannerImage.getImageHash() != null) {
                List<ModerationDiag> moderationDiags = bannerReasons.get(ModerationReasonObjectType.IMAGE);
                if (moderationDiags != null && !moderationDiags.isEmpty()) {
                    Map<String, Object> imageData = new LinkedHashMap<>();
                    imageData.put("reasons", convertList(moderationDiags));
                    imageData.put("name", bannerWithBannerImage.getImageName());
                    imageData.put("image_hash", bannerWithBannerImage.getImageHash());
                    Map<Long, Map<String, Object>> imageMap = ImmutableMap.of(banner.getId(), imageData);
                    moderationReportResponse.setImageMap(imageMap);
                }
            }
        }
    }

    private static void fillBadPhrases(ModerationReportResponse moderationReportResponse,
                                       BannerWithSystemFields banner,
                                       Map<Long, List<Keyword>> phrases) {
        if (phrases != null && !phrases.isEmpty()) {
            moderationReportResponse.setBadPhrases(badPhrasesConverter(phrases.get(banner.getAdGroupId())));
        }
    }
}
