package ru.yandex.direct.core.entity.moderationreason.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.SetUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithButton;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithLogo;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.model.ModerateBannerPage;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.repository.ModerateBannerPagesRepository;
import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService;
import ru.yandex.direct.core.entity.moderationreason.ModerationReasonFilterBuilder;
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReason;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonDetailed;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonKey;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonWithDetails;
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.collectors.FlipMapCollector;
import ru.yandex.direct.utils.collectors.MergeMapCollector;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.common.db.PpcPropertyNames.MODERATION_REASONS_ALLOWABLE_TO_REMODERATE_BY_CLIENT;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.BANNER;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.BANNER_BUTTON;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.BANNER_LOGO;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.BANNER_PAGE;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.CALLOUT;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.CANVAS;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.CONTACTINFO;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.DISPLAY_HREF;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.HTML5_CREATIVE;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.IMAGE;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.IMAGE_AD;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.PERF_CREATIVE;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.PHRASES;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.SITELINKS_SET;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.TURBOLANDING;
import static ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType.VIDEO_ADDITION;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class ModerationReasonService implements ModerationReasons {
    private static final Logger logger = LoggerFactory.getLogger(ModerationReasonService.class);

    private final ShardHelper shardHelper;
    private final ModerationReasonRepository moderationReasonRepository;
    private final ModerationDiagService moderationDiagService;
    private final BannerTypedRepository bannerTypedRepository;
    private final CreativeRepository creativeRepository;
    private final ModerateBannerPagesRepository moderateBannerPagesRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    public ModerationReasonService(ShardHelper shardHelper,
                                   ModerationReasonRepository moderationReasonRepository,
                                   ModerationDiagService moderationDiagService,
                                   BannerTypedRepository bannerTypedRepository,
                                   CreativeRepository creativeRepository,
                                   ModerateBannerPagesRepository moderateBannerPagesRepository,
                                   PpcPropertiesSupport ppcPropertiesSupport) {
        this.shardHelper = shardHelper;
        this.moderationReasonRepository = moderationReasonRepository;
        this.moderationDiagService = moderationDiagService;
        this.bannerTypedRepository = bannerTypedRepository;
        this.creativeRepository = creativeRepository;
        this.moderateBannerPagesRepository = moderateBannerPagesRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    private List<ModerationReason> getRejected(ClientId clientId, ModerationReasonObjectType objectType,
                                               Collection<Long> objectIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return moderationReasonRepository.fetchRejected(shard, objectType, objectIds);
    }

    public Map<Long, List<ModerationDiag>> getRejectReasonDiags(ClientId clientId,
                                                                ModerationReasonObjectType objectType,
                                                                Collection<Long> objectIds) {
        Map<Long, ModerationDiag> diags = moderationDiagService.get(ModerationDiagType.COMMON);
        List<ModerationReason> moderationReasons = getRejected(clientId, objectType, objectIds);
        return StreamEx.of(moderationReasons)
                .mapToEntry(ModerationReason::getObjectId, ModerationReason::getReasons)
                .mapValues(reasons -> StreamEx.of(reasons)
                        .map(ModerationReasonDetailed::getId).map(diags::get).nonNull().toList())
                .toMap();
    }

    public Map<Long, Map<BannerAssetType, List<ModerationDiag>>> getBannerAssetsRejectReasonDiags(
            ClientId clientId, List<Long> bannerIds) {
        return getBannerAssetsRejectReasonDiagsByShard(shardHelper.getShardByClientIdStrictly(clientId), bannerIds);
    }

    public Map<Long, Map<BannerAssetType, List<ModerationDiag>>> getBannerAssetsRejectReasonDiagsByShard(
            int shard, List<Long> bannerIds) {
        Map<Long, ModerationDiag> diags = moderationDiagService.get(ModerationDiagType.COMMON);
        List<ModerationReason> moderationReasons = moderationReasonRepository.fetchRejected(shard, BANNER, bannerIds);
        return StreamEx.of(moderationReasons)
                .mapToEntry(ModerationReason::getObjectId, ModerationReason::getAssetsReasons)
                .nonNullValues()
                .mapValues(reasonsByType -> EntryStream.of(reasonsByType)
                        .mapValues(reasons -> StreamEx.of(reasons)
                                .map(diags::get)
                                .nonNull()
                                .toList())
                        .toMap())
                .toMap();
    }

    public Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> getRejectReasonDiags(
            ClientId clientId,
            Map<ModerationReasonObjectType, List<Long>> objectIdsByObjectType) {
        return getRejectReasonDiagsByShard(shardHelper.getShardByClientIdStrictly(clientId), objectIdsByObjectType);
    }

    public Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> getRejectReasonDiagsByShard(
            int shard,
            Map<ModerationReasonObjectType, List<Long>> objectIdsByObjectType) {
        return getRejectReasonDiagsByShardRemapped(shard, EntryStream.of(objectIdsByObjectType)
                .mapValues(list -> StreamEx.of(list).toMap(Set::of)).toMap());
    }

    private Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> getRejectReasonDiagsByShardRemapped(
            int shard,
            Map<ModerationReasonObjectType, Map<Long, Set<Long>>> idsByObjectIdsByObjectType) {
        var diags = moderationDiagService.get(ModerationDiagType.COMMON);

        var rejectReasonIds = getRejectReasonIdsByShard(shard, idsByObjectIdsByObjectType);

        var hasRejectedPerfCreatives = EntryStream.of(rejectReasonIds)
                .values()
                .findAny(map -> map.get(PERF_CREATIVE) != null)
                .isPresent();

        if (hasRejectedPerfCreatives) {
            var creativeIdById = new HashMap<Long, Long>();
            for (var entry : idsByObjectIdsByObjectType.get(PERF_CREATIVE).entrySet()) {
                var creativeId = entry.getKey();
                for (var id : entry.getValue()) {
                    if (creativeIdById.get(id) != null) {
                        logger.error("tried to map perf creative '{}' to banner '{}' with mapped perf creative '{}'",
                                creativeId, id, creativeIdById.get(id));
                        throw new IllegalStateException("multiple perf creatives were mapped to a single banner");
                    }
                    creativeIdById.put(id, creativeId);
                }
            }

            var rejectedCreativeIds = EntryStream.of(rejectReasonIds)
                    .filterValues(map -> map.get(PERF_CREATIVE) != null)
                    .map(entry -> creativeIdById.get(entry.getKey()))
                    .toSet();
            var rejectedCreatives = creativeRepository.getCreatives(shard, rejectedCreativeIds);
            var bsModeratedCreatives = filterAndMapToSet(rejectedCreatives,
                    creative -> !Optional.of(creative)
                            .map(Creative::getAdditionalData)
                            .map(AdditionalData::getModeratedByDirect)
                            .orElse(false),
                    Creative::getId);
            if (!bsModeratedCreatives.isEmpty()) {
                var perfDiags = moderationDiagService.get(ModerationDiagType.PERFORMANCE);
                return EntryStream.of(rejectReasonIds)
                        .mapToValue((id, reasonIdsByObjectType) -> EntryStream.of(reasonIdsByObjectType)
                                .mapToValue((objectType, ids) -> StreamEx.of(ids)
                                        .map(objectType == PERF_CREATIVE
                                                && bsModeratedCreatives.contains(creativeIdById.get(id))
                                                ? perfDiags::get : diags::get)
                                        .nonNull()
                                        .toList())
                                .toMap())
                        .toMap();
            }
        }

        return EntryStream.of(rejectReasonIds)
                .mapValues(map -> EntryStream.of(map)
                        .mapValues(ids -> StreamEx.of(ids)
                                .map(diags::get)
                                .nonNull()
                                .toList())
                        .toMap())
                .toMap();
    }

    public Map<Long, Map<ModerationReasonObjectType, Set<Long>>> getRejectReasonIdsByShard(
            int shard,
            Map<ModerationReasonObjectType, Map<Long, Set<Long>>> idsByObjectIdsByObjectType) {
        var objectIdsByObjectType = EntryStream.of(idsByObjectIdsByObjectType)
                .mapValues(v -> List.copyOf(v.keySet())).toMap();
        var moderationReasons = moderationReasonRepository.fetchRejected(shard, objectIdsByObjectType);
        var result = new HashMap<Long, Map<ModerationReasonObjectType, Set<Long>>>();
        for (var moderationReason : moderationReasons) {
            var ids =
                    idsByObjectIdsByObjectType.get(moderationReason.getObjectType()).get(moderationReason.getObjectId());
            for (var id : ids) {
                var reasonIds = listToSet(moderationReason.getReasons(), ModerationReasonDetailed::getId);
                result.computeIfAbsent(id, k -> new EnumMap<>(ModerationReasonObjectType.class))
                        .merge(moderationReason.getObjectType(), reasonIds, (v1, v2) -> {
                            v1.addAll(v2);
                            return v1;
                        });
            }
        }
        return result;
    }

    public Map<ModerationReasonKey, List<ModerationDiag>> getRejectReasonDiagsByModerationKey(
            ClientId clientId, Set<ModerationReasonKey> moderationReasonKeys) {
        return getRejectReasonDiagsByModerationKey(shardHelper.getShardByClientIdStrictly(clientId),
                moderationReasonKeys);
    }

    private Map<ModerationReasonKey, List<ModerationDiag>> getRejectReasonDiagsByModerationKey(
            int shard, Set<ModerationReasonKey> moderationReasonKeys) {
        Map<Long, ModerationDiag> diags = moderationDiagService.get(ModerationDiagType.COMMON);
        Map<ModerationReasonObjectType, List<Long>> moderationReasonsByObjectTypeAndId =
                StreamEx.of(moderationReasonKeys)
                        .mapToEntry(ModerationReasonKey::getObjectType, ModerationReasonKey::getObjectId)
                        .distinct()
                        .grouping();
        List<ModerationReason> moderationReasons =
                moderationReasonRepository.fetchRejected(shard, moderationReasonsByObjectTypeAndId);

        Map<ModerationReasonKey, Set<Long>> diagIdsByKey = new HashMap<>();
        for (ModerationReason moderationReason : moderationReasons) {
            moderationReason.getReasons().forEach(reason -> {
                if (CollectionUtils.isEmpty(reason.getItemIds())) {
                    ModerationReasonKey moderationReasonKey = new ModerationReasonKey()
                            .withObjectId(moderationReason.getObjectId())
                            .withObjectType(moderationReason.getObjectType());
                    diagIdsByKey
                            .computeIfAbsent(moderationReasonKey, t -> new HashSet<>())
                            .add(reason.getId());
                } else {
                    reason.getItemIds().forEach(itemId -> {
                        ModerationReasonKey moderationReasonKey = new ModerationReasonKey()
                                .withObjectId(moderationReason.getObjectId())
                                .withObjectType(moderationReason.getObjectType())
                                .withSubObjectId(itemId);
                        diagIdsByKey
                                .computeIfAbsent(moderationReasonKey, t -> new HashSet<>())
                                .add(reason.getId());
                    });
                }
            });
        }
        return EntryStream.of(diagIdsByKey)
                .mapValues(diagIds -> mapList(diagIds, id -> {
                    if (!diags.containsKey(id)) {
                        logger.error("no reason for id= {} found", id);
                    }
                    return diags.get(id);
                }))
                .toMap();
    }

    private static final ModerationReasonFilterBuilder<?>[] FILTER_BUILDERS = {
            ModerationReasonFilterBuilder.simple(Banner.class, List.of(BANNER)),
            ModerationReasonFilterBuilder.withPredicate(BannerWithVcard.class, BannerWithVcard::getVcardId,
                    List.of(CONTACTINFO)),
            ModerationReasonFilterBuilder.withPredicate(BannerWithSitelinks.class,
                    BannerWithSitelinks::getSitelinksSetId,
                    List.of(SITELINKS_SET)),
            ModerationReasonFilterBuilder.simple(BannerWithBannerImage.class, List.of(IMAGE)),
            ModerationReasonFilterBuilder.withPredicate(BannerWithDisplayHref.class,
                    BannerWithDisplayHref::getDisplayHref,
                    List.of(DISPLAY_HREF)),
            ModerationReasonFilterBuilder.withPredicate(BannerWithTurboLanding.class,
                    BannerWithTurboLanding::getTurboLandingId,
                    List.of(TURBOLANDING)),
            ModerationReasonFilterBuilder.simple(BannerWithLogo.class, List.of(BANNER_LOGO)),
            ModerationReasonFilterBuilder.simple(BannerWithButton.class, List.of(BANNER_BUTTON)),
            ModerationReasonFilterBuilder.withMapper(BannerWithAdGroupId.class, BannerWithAdGroupId::getAdGroupId,
                    List.of(PHRASES)),
            ModerationReasonFilterBuilder.withMapper(BannerWithImage.class, BannerWithImage::getImageId,
                    List.of(IMAGE_AD)),
            ModerationReasonFilterBuilder.withMapper(BannerWithCreative.class,
                    BannerWithCreative::getCreativeRelationId,
                    List.of(CANVAS, HTML5_CREATIVE, VIDEO_ADDITION)),
            ModerationReasonFilterBuilder.withMapper(BannerWithCreative.class, BannerWithCreative::getCreativeId,
                    List.of(PERF_CREATIVE)),
            ModerationReasonFilterBuilder.withMultiMapper(BannerWithCallouts.class, BannerWithCallouts::getCalloutIds,
                    List.of(CALLOUT)),
    };

    private static final Set<Class<? extends Banner>> REQUIRED_BANNER_INTERFACES = StreamEx.of(FILTER_BUILDERS)
            .map(ModerationReasonFilterBuilder::getBannerClass)
            .collect(Collectors.toSet());

    private Map<ModerationReasonObjectType, Map<Long, Set<Long>>> getModerationReasonFilter(int shard,
                                                                                            List<Banner> banners) {
        Map<ModerationReasonObjectType, Map<Long, Set<Long>>> idsByObjectIdsByObjectTypes
                = new EnumMap<>(ModerationReasonObjectType.class);

        for (var filterBuilder : FILTER_BUILDERS) {
            filterBuilder.process(banners, idsByObjectIdsByObjectTypes);
        }

        var pagesByBannerIds = moderateBannerPagesRepository.getModerateBannerPagesByBannerIds(shard,
                StreamEx.of(banners).map(Banner::getId).toSet());
        Map<Long, Set<Long>> pages =
                EntryStream.of(pagesByBannerIds)
                        .values()
                        .flatMapToEntry(list -> StreamEx.of(list)
                                .mapToEntry(ModerateBannerPage::getPageId, ModerateBannerPage::getBannerId)
                                .toMap())
                        .groupingTo(HashSet::new);
        idsByObjectIdsByObjectTypes.merge(BANNER_PAGE, pages, ModerationReasonFilterBuilder::mergeMaps);

        return idsByObjectIdsByObjectTypes;
    }

    public List<ModerationReasonWithDetails> getReasonsWithDetails(int shard, Collection<Long> bannerIds) {
        final var banners = bannerTypedRepository.getSafely(shard, bannerIds, REQUIRED_BANNER_INTERFACES);
        final var bannersIdsByObjectIdByObjectType = getModerationReasonFilter(shard, banners);
        final var bannerIdsByObjId = bannersIdsByObjectIdByObjectType
                .values()
                .stream()
                .collect(new MergeMapCollector<>())
                .entrySet()
                .stream()
                .collect(new FlipMapCollector<>());

        final Map<ModerationReasonObjectType, List<Long>> objectIdsByObjectType = bannersIdsByObjectIdByObjectType
                .entrySet()
                .stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> new ArrayList<>(e.getValue().keySet())));

        final var diags = moderationDiagService.get(ModerationDiagType.COMMON);

        return moderationReasonRepository.fetchRejected(shard, objectIdsByObjectType)
                .stream()
                .flatMap(modReason -> ModerationReasonWithDetails.fromModerationReason(modReason, diags,
                        bannerIdsByObjId.get(modReason.getObjectId())).stream())
                .collect(Collectors.toList());
    }

    public Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> getReasons(Collection<Long> bannerIds) {
        Map<Integer, List<Banner>> banners = shardHelper.groupByShard(bannerIds, ShardKey.BID)
                .stream().filter(Objects::nonNull).collect(Collectors
                        .toMap(Map.Entry::getKey, e ->
                                bannerTypedRepository.getSafely(e.getKey(), e.getValue(), REQUIRED_BANNER_INTERFACES)));

        return getReasons(banners);
    }

    public Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> getReasons(int shard,
                                                                                       Collection<Long> bannerIds) {
        Map<Integer, List<Banner>> banners =
                Map.of(shard, bannerTypedRepository.getSafely(shard, bannerIds, REQUIRED_BANNER_INTERFACES));

        return getReasons(banners);
    }

    public Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> getReasons(
            Map<Integer, List<Banner>> bannersByShard) {

        return EntryStream.of(bannersByShard)
                .flatMapKeyValue((k, v) -> EntryStream.of(getRejectReasonDiagsByShardRemapped(k,
                        getModerationReasonFilter(k, v))))
                .toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    public List<ModerationDiag> getModerationDiagsByIds(List<Long> diagIds) {
        Map<Long, ModerationDiag> diags = moderationDiagService.get(ModerationDiagType.PERFORMANCE);
        return diagIds.stream().map(diags::get).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public Map<Long, List<ModerationDiag>> getCalloutRejectedReasons(Long bannerId, List<Long> objectIds) {
        return EntryStream.of(getRejectReasonDiagsByShard(shardHelper.getShardByBannerId(bannerId),
                        ImmutableMap.of(CALLOUT, objectIds)))
                .mapValues(v -> v.get(CALLOUT)).toMap();
    }

    public Map<Long, Set<Long>> getReasonIdsForBannerAndResources(int shard, Set<Long> adIds) {
        //Превращаем в соответствие баннер — все причины отклонения
        var moderationReasonsByObjectTypeAndId = getReasons(shard, adIds);
        return EntryStream.of(moderationReasonsByObjectTypeAndId)
                .mapValues(moderationReasonObjectTypeListMap ->
                        EntryStream.of(moderationReasonObjectTypeListMap)
                                .mapValues(moderationDiags -> StreamEx.of(moderationDiags)
                                        .map(ModerationDiag::getId)
                                )
                                .values()
                                .flatMap(Function.identity())
                                .toSet()
                ).toMap();
    }

    public Set<Long> filterReasonsNonAllowedToSelfRemoderate(Set<Long> reasons) {
        var allowedReasonsToRemoderate = ppcPropertiesSupport
                .get(MODERATION_REASONS_ALLOWABLE_TO_REMODERATE_BY_CLIENT)
                .getOrDefault(emptySet());
        return filterReasonsNonAllowedToSelfRemoderate(reasons, allowedReasonsToRemoderate);
    }

    public static Set<Long> filterReasonsNonAllowedToSelfRemoderate(Set<Long> actualReasons,
                                                                    Set<Long> allowedReasonsToRemoderate) {
        return SetUtils.difference(actualReasons, allowedReasonsToRemoderate);
    }

    public static Predicate<Set<Long>> allReasonsAllowedSelfRemoderate(Set<Long> allowedReasonsToRemoderate) {
        return reasons -> filterReasonsNonAllowedToSelfRemoderate(reasons, allowedReasonsToRemoderate).isEmpty();
    }
}
