package ru.yandex.direct.core.entity.moderationreason.service;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JsonNode;
import org.apache.commons.lang3.tuple.Pair;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.AsyncHttpClientConfig;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.JsonParsableRequest;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.moderationdiag.DocCenterTooltipModel;
import ru.yandex.direct.core.entity.moderationdiag.DocCenterTooltipModelSelector;
import ru.yandex.direct.core.entity.moderationdiag.DocCenterTooltipParsedData;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonDescription;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonRequest;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonsResponse;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Service
public class ModerationReasonTextService implements ModerationReasonText {
    private static final Logger logger = LoggerFactory.getLogger(ModerationReasonTextService.class);

    private final ParallelFetcherFactory parallelFetcherFactory;
    private final CreativeRepository creativeRepository;
    private final ShardHelper shardHelper;
    private final ModerationDiagService moderationDiagService;


    private final String reasonTooltipURL;

    private static final Map<Language, String> langToTldMap =
            Map.of(
                    Language.RU, "ru",
                    Language.UK, "ua",
                    Language.EN, "com",
                    Language.TR, "com.tr"
            );

    private static final Map<String, List<String>> tldPriority = new HashMap<>();

    static {
        tldPriority.put("ru", Collections.singletonList("ru"));
        tldPriority.put("by", Arrays.asList("by", "ru"));
        tldPriority.put("ua", Arrays.asList("ua", "ru", "en"));
        tldPriority.put("com", Arrays.asList("com", "ru"));
        tldPriority.put("com.tr", Arrays.asList("com.tr", "com", "ru"));
        tldPriority.put("kz", Arrays.asList("kz", "ru", "com"));
    }

    @Autowired
    public ModerationReasonTextService(
            @Value("${moderation.urls.reason-tooltip}") String reasonTooltipURL,
            AsyncHttpClient asyncHttpClient,
            CreativeRepository creativeRepository,
            ShardHelper shardHelper,
            ModerationDiagService moderationDiagService) {
        this.reasonTooltipURL = reasonTooltipURL;
        this.shardHelper = shardHelper;
        this.moderationDiagService = moderationDiagService;

        AsyncHttpClientConfig config = asyncHttpClient.getConfig();
        FetcherSettings fetcherSettings = new FetcherSettings()
                .withParallel(config.getIoThreadsCount())
                .withConnectTimeout(Duration.ofMillis(config.getConnectTimeout()))
                .withGlobalTimeout(Duration.ofMillis(config.getRequestTimeout()))
                .withRequestTimeout(Duration.ofMillis(config.getRequestTimeout()))
                .withRequestRetries(config.getMaxRequestRetry());

        this.parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, fetcherSettings);
        this.creativeRepository = creativeRepository;
    }

    private boolean isModeratedByBannerStorage(Creative creative) {
        if (creative.getType() != CreativeType.PERFORMANCE) {
            return false;
        }
        AdditionalData additionalData = creative.getAdditionalData();
        return additionalData == null
                || additionalData.getModeratedByDirect() == null
                || !additionalData.getModeratedByDirect();
    }

    private String wrapTitle(String title) {
        return String.format("<h1 class=\\\"title topictitle1 doc-c-headers doc-c-headers_mod_h1\\\" " +
                "id=\\\"ariaid-title1\\\" data-help-title=\\\"1\\\">%s</h1>", title);
    }

    private String wrapContent(Long id, String title, String content) {
        return String.format("<main role=\"main\" class=\"doc-c doc-c-main doc-c-i-bem\" doc-data-bem=\"{&quot;" +
                "doc-c-main&quot;:{}}\">\n" +
                "    <article role=\"article\" aria-labelledby=\"ariaid-title1\" class=\"doc-c-article\"><h1\n" +
                "            class=\"doc-c-title doc-c-topictitle1 doc-c-headers doc-c-headers_mod_h1\" " +
                "id=\"ariaid-title1\"\n" +
                "            data-help-title=\"1\">%s</h1>\n" +
                "        <div class=\"doc-c-body doc-c-conbody\" data-help-text=\"1\">\n" +
                "            <section class=\"doc-c-section\" id=\"modadvert-%d__problem\">%s</section>\n" +
                "        </div>\n" +
                "    </article>\n" +
                "</main>", title, id, content);
    }

    /**
     * Возвращает тексты причин отклонений, пригодные для вывода во всплывающем окне статуса объявления
     * Обычно bannerId не указан, и тексты берутся из ДокЦентра (поведение по умолчанию)
     * Если же bannerId указан, то сначала будет проверка, не связан ли баннер с performance-креативом,
     * который модерировался старой премодерацией внутри BannerStorage. И если это так, то тексты причин
     * получаются иначе: из таблицы ppcdict.moderate_diags с типом performance
     * (иначе тексты причин будут неправильные: номера причин с типом common и performance соответствуют разным
     * причинам)
     * В будущем, надеюсь, мы уберём эти legacy-причины, смигрировав их в общий тип common
     * (видимо, на другое множество айдишников), чтобы можно было выпилить эти костыли
     */
    public ModerationReasonsResponse showModReasons(@Nonnull ModerationReasonRequest request,
                                                    @Nullable Long bannerId) {
        if (bannerId != null) {
            int shard = shardHelper.getShardByBannerId(bannerId);
            Creative creative = creativeRepository.getCreativesByBannerIds(shard, singleton(bannerId)).get(bannerId);
            if (creative != null && isModeratedByBannerStorage(creative)) {
                Map<Long, ModerationDiag> diagMap = moderationDiagService.get(ModerationDiagType.PERFORMANCE);
                Map<String, String> reasonsMap = Arrays.stream(request.getTooltipIDs())
                        .map(tooltipId -> {
                            Long modReasonId = Long.parseLong(tooltipId);
                            ModerationDiag diag = diagMap.get(modReasonId);
                            String title = diag.getShortText();
                            String content = diag.getFullText();
                            var tooltipModel = new DocCenterTooltipModel(
                                    wrapTitle(title),
                                    new DocCenterTooltipModelSelector(wrapContent(modReasonId, title, content))
                            );
                            String json = toJson(tooltipModel);
                            return Pair.of(tooltipId, json);
                        })
                        .collect(toMap(Pair::getKey, Pair::getValue));
                return new ModerationReasonsResponse(reasonsMap, emptyList());
            }
        }

        if (request.getTooltipIDs() == null || request.getTooltipIDs().length == 0) {
            return new ModerationReasonsResponse(Collections.emptyMap(), Collections.emptyList());
        }

        final var trueTld = request.getLang() != null ? langToTldMap.get(request.getLang()) : request.getTld();
        final List<Tuple3<String, String, String>> toFetch =
                Arrays.stream(request.getTooltipIDs()).map(id -> Tuple3.tuple(id, trueTld, trueTld))
                        .collect(Collectors.toList());

        try {
            Map<String, String> success = new HashMap<>();
            List<String> errors = new ArrayList<>();

            while (!toFetch.isEmpty()) {
                Tuple2<Map<String, String>, List<Tuple3<String, String, String>>> res = fetchTooltips(toFetch);

                success.putAll(res.get1());

                if (res.get2().isEmpty()) {
                    break;
                }

                toFetch.clear();

                res.get2().forEach(error -> {
                    String nextTld = nextTld(error.get2(), error.get3());

                    if (nextTld == null) {
                        errors.add(error.get1());
                    } else {
                        toFetch.add(Tuple3.tuple(error.get1(), error.get2(), nextTld));
                    }
                });
            }

            return new ModerationReasonsResponse(success, errors);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    private String nextTld(String initTld, String currentTld) {
        if (!tldPriority.containsKey(initTld)) {
            return null;
        }

        List<String> tlds = tldPriority.get(initTld);

        int i = tlds.indexOf(currentTld);

        if (i == -1 || i == tlds.size() - 1) {
            return null;
        }

        return tlds.get(i + 1);
    }

    private Tuple2<Map<String, String>, List<Tuple3<String, String, String>>> fetchTooltips(
            List<Tuple3<String, String, String>> tooltips)
            throws InterruptedException {
        Map<Long, Tuple3<String, String, String>> id2string = new HashMap<>();
        long[] idGenerator = {0};

        List<JsonParsableRequest<ModerationReasonDescription>> requests = tooltips.stream()
                .map(tooltip -> {
                    long requestID = idGenerator[0]++;

                    id2string.put(requestID, tooltip);

                    return new JsonParsableRequest<>(requestID, request(tooltip), ModerationReasonDescription.class);
                })
                .collect(Collectors.toList());

        ParallelFetcher<ModerationReasonDescription> parallelFetcher = parallelFetcherFactory.getParallelFetcher();

        Map<Long, Result<ModerationReasonDescription>> requestResults = parallelFetcher.execute(requests);

        Map<String, String> response = new HashMap<>();
        List<Tuple3<String, String, String>> errors = new ArrayList<>();

        requestResults.forEach((id, result) -> {
            if (result.getSuccess() != null) {
                JsonNode json = fromJson(result.getSuccess().getDoccenterResponse());
                final var parsedReasonData = new DocCenterTooltipParsedData(json);

                response.put(id2string.get(id).get1(), parsedReasonData.json());
            } else {
                result.getErrors().forEach(e -> logger.warn("Got error while fetching moderation reasons", e));
                errors.add(id2string.get(id));
            }
        });

        return Tuple2.tuple(response, errors);
    }

    private Request request(Tuple3<String, String, String> tooltip) {
        String body =
                ("{\"tld\": \"{tld}\", \"doccenterParams\": {\"forceFullLinks\": true, \"components\": [\"content\", " +
                        "\"selector(query=.title)\"]}}")
                        .replace("{tld}", tooltip.get3());

        String url = reasonTooltipURL
                .replace("{id}", tooltip.get1())
                .replace("{tld}", tooltip.get3());

        return new RequestBuilder("POST")
                .setUrl(url)
                .addHeader("Content-Type", "application/json")
                .setBody(body)
                .build();

    }
}
