package ru.yandex.direct.core.entity.motivationmail;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import org.jooq.Field;
import org.jooq.Record7;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.ClientReminderLettersLastSentNotification;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.utils.FunctionalUtils;

import static ru.yandex.direct.dbschema.ppc.tables.CampOptions.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.ClientReminderLetters.CLIENT_REMINDER_LETTERS;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsOptions.CLIENTS_OPTIONS;

/**
 * Репозиторий для работы с мотивационными данными новых клиентов
 */
@Repository
public class MotivationMailRepository {
    private final DslContextProvider dslContextProvider;

    private static final Collection<CampaignsType> CAMPAIGNS_OF_INTEREST =
            FunctionalUtils.mapList(CampaignTypeKinds.WEB_EDIT_BASE, CampaignType::toSource);

    private static final Field<Integer> newCampsCount = DSL.count(DSL
            .when(CAMPAIGNS.STATUS_MODERATE.eq(CampaignsStatusmoderate.New)
                    .and(CAMPAIGNS.TYPE.in(CAMPAIGNS_OF_INTEREST)), 1)
            .otherwise(DSL.castNull(Integer.class)))
            .as("new_camps_count");

    private static final Field<Integer> acceptedCampsCount = DSL.count(DSL
            .when(CAMPAIGNS.STATUS_MODERATE.eq(CampaignsStatusmoderate.Yes)
                    .and(CAMP_OPTIONS.STATUS_POST_MODERATE.eq(CampOptionsStatuspostmoderate.Accepted))
                    .and(CAMPAIGNS.TYPE.in(CAMPAIGNS_OF_INTEREST)), 1)
            .otherwise(DSL.castNull(Integer.class)))
            .as("accepted_camps_count");

    private static final Field<Integer> totalSum = DSL.count(DSL.when(CAMPAIGNS.SUM.gt(BigDecimal.ZERO), 1)
            .otherwise(DSL.castNull(Integer.class)))
            .as("total_sum");

    // важно, чтобы строчки TOUCH и DESKTOP были написаны большими буквами, т.к. они используются для конвертации в
    // Enum вот так: MotivationMailStats.ClientType.valueOf(record.get(clientType))
    private static final Field<String> clientType = DSL.field(
            "IF(FIND_IN_SET('is_touch', clients_options.client_flags), 'TOUCH', 'DESKTOP')", String.class)
            .as("clientType");

    @Autowired
    public MotivationMailRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * возвращает поклиентскую статистику для шарда {@code shard}
     */
    public List<MotivationMailStats> getAll(int shard) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_REMINDER_LETTERS.CLIENT_ID, CLIENT_REMINDER_LETTERS.LAST_SENT_NOTIFICATION,
                        CLIENT_REMINDER_LETTERS.LAST_SENT_TIME, newCampsCount, acceptedCampsCount, totalSum, clientType)
                .from(CLIENT_REMINDER_LETTERS)
                .leftJoin(CAMPAIGNS).on(CAMPAIGNS.CLIENT_ID.eq(CLIENT_REMINDER_LETTERS.CLIENT_ID))
                .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                // если захотите давать таблице clients_options какой-то алиас, поменяйте определение поля clientType
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CLIENT_REMINDER_LETTERS.CLIENT_ID))
                .where(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No).or(CAMPAIGNS.CID.isNull()))
                .groupBy(CLIENT_REMINDER_LETTERS.CLIENT_ID)
                .fetch(MotivationMailRepository::motivationMailStatMapper);
    }

    private static MotivationMailStats motivationMailStatMapper(
            Record7<Long, ClientReminderLettersLastSentNotification, LocalDateTime, Integer, Integer, Integer, String> record) {
        return new MotivationMailStats(record.get(CLIENT_REMINDER_LETTERS.CLIENT_ID),
                MotivationMailRepositoryMappings.motivationMailNotificationTypeFromDb(
                        record.get(CLIENT_REMINDER_LETTERS.LAST_SENT_NOTIFICATION)),
                record.get(CLIENT_REMINDER_LETTERS.LAST_SENT_TIME),
                record.get(newCampsCount),
                record.get(acceptedCampsCount),
                record.get(totalSum),
                MotivationMailStats.ClientType.valueOf(record.get(clientType)));
    }

    /**
     * обновляет время и тип последнего мотивационного письма на клиента
     */
    public void updateNotificationInfo(int shard, long clientId, MotivationMailNotificationType notification,
                                       LocalDateTime notificationTime) {
        dslContextProvider.ppc(shard)
                .update(CLIENT_REMINDER_LETTERS)
                .set(CLIENT_REMINDER_LETTERS.LAST_SENT_NOTIFICATION,
                        MotivationMailRepositoryMappings.motivationMailNotificationTypeToDb(notification))
                .set(CLIENT_REMINDER_LETTERS.LAST_SENT_TIME, notificationTime)
                .where(CLIENT_REMINDER_LETTERS.CLIENT_ID.eq(clientId))
                .execute();
    }

    /**
     * Добавляет данные о мотивационных письмах клиента {@code clientId} в шарде {@code shard}
     */
    public void addNotificationInfo(int shard, long clientId, MotivationMailNotificationType notification,
                                    LocalDateTime notificationTime) {
        dslContextProvider.ppc(shard)
                .insertInto(CLIENT_REMINDER_LETTERS, CLIENT_REMINDER_LETTERS.CLIENT_ID,
                        CLIENT_REMINDER_LETTERS.LAST_SENT_NOTIFICATION, CLIENT_REMINDER_LETTERS.LAST_SENT_TIME)
                .values(clientId, MotivationMailRepositoryMappings.motivationMailNotificationTypeToDb(notification),
                        notificationTime)
                .execute();
    }

    /**
     * Удаляет данные о мотивационных письмах клиента {@code clientId} в шарде {@code shard}
     */
    public void deleteNotificationInfo(int shard, long clientId) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENT_REMINDER_LETTERS)
                .where(CLIENT_REMINDER_LETTERS.CLIENT_ID.eq(clientId))
                .execute();
    }
}
