package ru.yandex.direct.core.entity.notification.container;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;

import static java.util.Objects.requireNonNull;

public class MobileContentMonitoringNotification implements Notification {
    private final long uid;
    private final String fio;
    private final State state;
    private final Collection<Record> stateChanges;

    public MobileContentMonitoringNotification(long uid, String fio,
                                               State state,
                                               Collection<Record> stateChanges) {
        this.uid = uid;
        this.fio = fio;
        this.state = requireNonNull(state);
        this.stateChanges = requireNonNull(stateChanges);
    }

    public void addStateChangeRecord(Record record) {
        stateChanges.add(record);
    }

    @Override
    public NotificationType getNotificationType() {
        return NotificationType.MOBILE_CONTENT_MONITORING;
    }

    @Override
    public Map<String, Object> getNotificationData() {
        Map<String, Object> data = new HashMap<>();
        data.put("uid", uid);
        data.put("FIO", fio);
        data.put("state", state);
        data.put("changes", stateChanges);

        return Collections.unmodifiableMap(data);
    }

    public enum State {
        DEAD,
        ALIVE,
        ;

        @JsonCreator
        public static State parse(String s) {
            return State.valueOf(s.toUpperCase());
        }

        @JsonValue
        public String value() {
            return this.name().toLowerCase();
        }
    }

    public static class Record {
        @JsonProperty("mobile_content_id")
        private final long mobileContentId;
        @JsonProperty("uid")
        private final long uid;
        @JsonProperty("FIO")
        private final String fio;
        @JsonProperty("cid")
        private final long campaignId;
        @JsonProperty("store_content_href")
        private final String storeContentHref;

        private Record(long mobileContentId, long uid, String fio, long campaignId, String storeContentHref) {
            this.mobileContentId = mobileContentId;
            this.uid = uid;
            this.fio = fio;
            this.campaignId = campaignId;
            this.storeContentHref = storeContentHref;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static final class Builder {
            private long mobileContentId;
            private long uid;
            private String fio;
            private long campaignId;

            private String storeContentHref;

            private Builder() {
            }

            public Builder withMobileContentId(long mobileContentId) {
                this.mobileContentId = mobileContentId;
                return this;
            }

            public Builder withUid(long uid) {
                this.uid = uid;
                return this;
            }

            public Builder withFio(String fio) {
                this.fio = fio;
                return this;
            }

            public Builder withCampaignId(long campaignId) {
                this.campaignId = campaignId;
                return this;
            }

            public Builder withStoreContentHref(String storeContentHref) {
                this.storeContentHref = storeContentHref;
                return this;
            }

            public Record build() {
                return new Record(mobileContentId, uid, fio, campaignId, storeContentHref);
            }
        }
    }
}
