package ru.yandex.direct.core.entity.offerretargeting.container

import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.offerretargeting.service.OfferRetargetingAddOperation
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.dbutil.model.ClientId

/**
 * Абстрактный класс контейнера для операции добавления офферных ретаргетингов
 * @param operatorUid идентификатор пользователя, являющегося инициатором операции
 * @param clientId идентификатор клиента, для которого выполняется операция
 * @param clientCurrency валюта клиента, для которого выполняется операция
 * @see OfferRetargetingAddOperation
 */
sealed class OfferRetargetingAddContainer(
    open val operatorUid: Long,
    open val clientId: ClientId,
    open val clientCurrency: Currency
) {
    abstract fun getCampaign(offerRetargetingIndex: Int?, adGroupId: Long?): Campaign?
}

/**
 * Контейнер для операции добавления офферных ретаргетингов с существующими группами
 * @param campaignsByIds кампании клиента по id
 * @param campaignIdsByAdGroupIds id кампаний по id групп клиента, к которым добавляется офферный ретаргетинг
 * @see OfferRetargetingAddOperation
 */
data class OfferRetargetingAddContainerWithExistentAdGroups(
    override val operatorUid: Long,
    override val clientId: ClientId,
    override val clientCurrency: Currency,
    val campaignsByIds: Map<Long, Campaign>,
    val campaignIdsByAdGroupIds: Map<Long, Long>
) : OfferRetargetingAddContainer(operatorUid, clientId, clientCurrency) {
    override fun getCampaign(offerRetargetingIndex: Int?, adGroupId: Long?): Campaign? =
        getCampaignByAdGroupId(requireNotNull(adGroupId))

    fun getCampaignByAdGroupId(adGroupId: Long): Campaign? =
        campaignIdsByAdGroupIds[adGroupId]?.let { campaignsByIds[it] }
}

/**
 * Информация о группе, которая будет создана в рамках связанной операцией добавления групп
 * @param adGroupIndex индекс группы в списке создаваемых групп в связанной операции добавления групп
 * @param campaign кампания, для которой добавляется данная группа
 * @see OfferRetargetingAddContainerWithNonExistentAdGroups
 */
data class AdGroupInfoForOfferRetargetingAdd(
    val adGroupIndex: Int,
    val campaign: Campaign
)

/**
 * Контейнер для операции добавления офферных ретаргетингов, которая будет выполняться в рамках операции создания групп
 * @param adGroupInfos информация о создаваемых группах по индексу офферного ретаргетинга
 * в списке [OfferRetargetingAddOperation.models]
 * @see OfferRetargetingAddOperation
 */
data class OfferRetargetingAddContainerWithNonExistentAdGroups(
    override val operatorUid: Long,
    override val clientId: ClientId,
    override val clientCurrency: Currency,
    var adGroupInfos: Map<Int, AdGroupInfoForOfferRetargetingAdd> = emptyMap()
) : OfferRetargetingAddContainer(operatorUid, clientId, clientCurrency) {
    override fun getCampaign(offerRetargetingIndex: Int?, adGroupId: Long?): Campaign? =
        getCampaignByOfferRetargetingIndex(requireNotNull(offerRetargetingIndex))

    /**
     * Проверка того, что информация о создаваемых группах выставлена для всех создаваемых офферных ретаргетингов
     * @param offerRetargetingsSize количество создаваемых офферных ретаргетингов
     */
    fun checkConsistency(offerRetargetingsSize: Int) {
        require((0 until offerRetargetingsSize).all { it in adGroupInfos })
    }

    fun getCampaignByOfferRetargetingIndex(offerRetargetingIndex: Int): Campaign? =
        adGroupInfos[offerRetargetingIndex]?.campaign
}
