package ru.yandex.direct.core.entity.offerretargeting.container

import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.service.OfferRetargetingUpdateOperation
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.dbutil.model.ClientId

/**
 * Контейнер для операции обновления офферных ретаргетингов
 * @param operatorUid - идентификатор пользователя, являющегося инициатором операции
 * @param clientId - идентификатор клиента, для которого выполняется операция
 * @param clientUid -
 * @param clientCurrency - валюта клиента, для которого выполняется операция
 * @param campaignsByIds - кампании клиента по id
 * @param campaignIdsByAdGroupIds - id кампаний по id групп клиента, в которых обновляется офферный ретаргетинг
 * @param adGroupIdsByOfferRetargetingIds - id групп по id офферных ретаргетингов клиента
 * @param offerRetargetingsByIds - офферные ретаргетинга клиента по id
 * @see OfferRetargetingUpdateOperation
 */
data class OfferRetargetingUpdateContainer(
    val operatorUid: Long,
    val clientId: ClientId,
    val clientUid: Long,
    val clientCurrency: Currency,
    val campaignsByIds: Map<Long, Campaign>,
    val campaignIdsByAdGroupIds: Map<Long, Long>,
    val adGroupIdsByOfferRetargetingIds: Map<Long, Long>,
    val offerRetargetingsByIds: Map<Long, OfferRetargeting>
) {
    init {
        /**
         * Так как для операции обновления клиент не выставляет id кампании и id группы, его выставляем мы.
         * Делаем проверку, что данные консистентны и кампании находятся по идентификаторам.
         */
        require(offerRetargetingsByIds.keys.all { it in adGroupIdsByOfferRetargetingIds })
        require(offerRetargetingsByIds.keys.all { adGroupIdsByOfferRetargetingIds[it] in campaignIdsByAdGroupIds })
        require(offerRetargetingsByIds.keys.all {
            val adGroupId = adGroupIdsByOfferRetargetingIds[it]
            campaignIdsByAdGroupIds[adGroupId] in campaignsByIds
        })
    }

    fun getCampaignByOfferRetargetingId(offerRetargetingId: Long): Campaign {
        val adGroupId = adGroupIdsByOfferRetargetingIds[offerRetargetingId]
        val campaignId = campaignIdsByAdGroupIds[adGroupId]
        return campaignsByIds[campaignId]!!
    }

    fun getCampaignById(campaignId: Long): Campaign {
        return campaignsByIds[campaignId]!!
    }
}
