package ru.yandex.direct.core.entity.offerretargeting.repository

import one.util.streamex.StreamEx
import org.jooq.Configuration
import org.jooq.Field
import org.jooq.impl.DSL
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria
import ru.yandex.direct.core.entity.bids.repository.BidMappings
import ru.yandex.direct.core.entity.bids.service.BidBaseOpt
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingMapping.offerRetargetingPropsToDbOpts
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingMapping.offerRetargetingToCoreModelChanges
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingMapping.priceFromDbFormat
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingMapping.priceToDbFormat
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseStatusbssynced
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty
import ru.yandex.direct.dbschema.ppc.tables.BidsBase
import ru.yandex.direct.dbschema.ppc.tables.Campaigns
import ru.yandex.direct.dbschema.ppc.tables.Phrases
import ru.yandex.direct.dbutil.SqlUtils
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property
import ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField
import ru.yandex.direct.jooqmapper.write.PropertyValues
import ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperties
import ru.yandex.direct.jooqmapper.write.WriterBuilders.fromSupplier
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelProperty
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.operation.AddedModelId

@Repository
class OfferRetargetingRepository @Autowired constructor(
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper
) {
    private val jooqMapper: JooqMapperWithSupplier<OfferRetargeting>

    init {
        jooqMapper = buildBidJooqMapper()
    }

    private fun buildBidJooqMapper(): JooqMapperWithSupplier<OfferRetargeting> =
        JooqMapperWithSupplierBuilder.builder { OfferRetargeting() }
            .map(property(OfferRetargeting.ID, BidsBase.BIDS_BASE.BID_ID))
            .map(property(OfferRetargeting.CAMPAIGN_ID, BidsBase.BIDS_BASE.CID))
            .map(property(OfferRetargeting.AD_GROUP_ID, BidsBase.BIDS_BASE.PID))
            .writeField(Tables.BIDS_BASE.BID_TYPE, fromSupplier { BidsBaseBidType.offer_retargeting })
            .map(convertibleProperty(OfferRetargeting.PRICE, BidsBase.BIDS_BASE.PRICE, ::priceFromDbFormat, ::priceToDbFormat))
            .map(
                convertibleProperty(
                    OfferRetargeting.PRICE_CONTEXT,
                    BidsBase.BIDS_BASE.PRICE_CONTEXT,
                    ::priceFromDbFormat,
                    ::priceToDbFormat
                )
            )
            .map(property(OfferRetargeting.LAST_CHANGE_TIME, BidsBase.BIDS_BASE.LAST_CHANGE))
            .map(
                convertibleProperty(
                    OfferRetargeting.STATUS_BS_SYNCED, BidsBase.BIDS_BASE.STATUS_BS_SYNCED,
                    BidMappings::statusBsSyncedFromDbFormat, BidMappings::statusBsSyncedToDbFormat
                )
            )
            .readProperty(
                OfferRetargeting.IS_SUSPENDED,
                fromField(BidsBase.BIDS_BASE.OPTS).by(BidMappings::isSuspendedFromDbOpts)
            )
            .readProperty(OfferRetargeting.IS_DELETED, fromField(BidsBase.BIDS_BASE.OPTS).by(BidMappings::isDeletedFromDbOpts))
            .writeField(
                BidsBase.BIDS_BASE.OPTS,
                fromProperties(
                    OfferRetargeting.IS_SUSPENDED,
                    OfferRetargeting.IS_DELETED
                ).by(BidMappings::bidPropsToDbOpts)
            )
            .build()

    /**
     * При добавлении нового офферного ретаргетинга проверяем:
     * если в группе есть офферный ретаргетинг и он удален, восстанавливаем его, с новыми
     * настройками и старым id,
     * если в группе есть офферный ретаргетинг и он не удален, ничего не меняем, прокидываем старый id,
     * иначе добавляем новый офферный ретаргетинг.
     */
    fun addOfferRetargetings(
        config: Configuration, clientId: ClientId,
        offerRetargetings: List<OfferRetargeting>,
        affectedAdGroupIds: Set<Long>
    ): List<AddedModelId> {
        val offerRetargetingsWithDeletedByAdGroupIds =
            getOfferRetargetingsWithDeletedByAdGroupIds(config, clientId, affectedAdGroupIds)
        val toAdd = mutableListOf<OfferRetargeting>()
        val toUpdate = mutableListOf<OfferRetargeting>()
        val result = mutableListOf<AddedModelId>()

        for (offerRetargeting in offerRetargetings) {
            val existingOfferRetargeting = offerRetargetingsWithDeletedByAdGroupIds[offerRetargeting.adGroupId]
            if (existingOfferRetargeting != null) {
                // offer retargeting exists
                offerRetargeting.id = existingOfferRetargeting.id
                if (existingOfferRetargeting.isDeleted) {
                    toUpdate.add(offerRetargeting)
                    result.add(AddedModelId.ofNew(existingOfferRetargeting.id))
                } else {
                    result.add(AddedModelId.ofExisting(existingOfferRetargeting.id))
                }
            } else {
                val newOfferRetargetingId = shardHelper.generatePhraseIds(1)[0]
                offerRetargeting.id = newOfferRetargetingId
                toAdd.add(offerRetargeting)
                result.add(AddedModelId.ofNew(newOfferRetargetingId))
            }
        }

        addNewOfferRetargetings(config, toAdd)
        updateDeletedOfferRetargeting(config, toUpdate)
        return result
    }

    /**
     * Добавление офферного ретаргетинга в группу, в которой нет этого офферного ретаргетинга.
     */
    private fun addNewOfferRetargetings(config: Configuration, offerRetargetings: List<OfferRetargeting>) {
        InsertHelper(DSL.using(config), BidsBase.BIDS_BASE)
            .addAll(jooqMapper, offerRetargetings)
            .executeIfRecordsAdded()
    }

    /**
     * восстанавливаем удаленный офферный ретаргетинг, с новыми настройками
     */
    private fun updateDeletedOfferRetargeting(config: Configuration, offerRetargetings: List<OfferRetargeting>) {
        val appliedChanges = offerRetargetings
            .associateWith { offerRetargetingToCoreModelChanges(it) }
            .map { (rm, changes) ->
                changes.applyTo( // изменения на офферный ретаргетинг со старыми идентфикаторами
                    OfferRetargeting()
                        .withId(rm.id)
                        .withAdGroupId(rm.adGroupId)
                        .withCampaignId(rm.campaignId)
                )
            }
        update(config, appliedChanges)
    }

    /**
     * метод используется при добавлении нового офферного ретаргетинга.
     * на текущий момент, у группы может быть только один офферный ретаргетинг. логики для добавления нескольких бт нет.
     *
     * @return возвращает офферные ретаргетинги клиента, принадлежащие переданным группам,
     * включая офферный ретаргетинг с флагом isDeleted = true
     */
    private fun getOfferRetargetingsWithDeletedByAdGroupIds(
        config: Configuration,
        clientId: ClientId,
        adGroupIds: Collection<Long>
    ): Map<Long, OfferRetargeting> {
        val offerRetargetings = config.dsl()
            .select(jooqMapper.fieldsToRead)
            .from(BidsBase.BIDS_BASE)
            .join(Campaigns.CAMPAIGNS).on(Campaigns.CAMPAIGNS.CID.eq(BidsBase.BIDS_BASE.CID))
            .where(BidsBase.BIDS_BASE.PID.`in`(adGroupIds))
            .and(BidsBase.BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
            .and(Campaigns.CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
            .fetch(jooqMapper::fromDb)
        return offerRetargetings.associateBy { it.adGroupId }
    }

    /**
     * @return возвращает офферный ретаргетинг клиента по передеанным id,
     * игнорируя офферный ретаргетинг с флагом isDeleted = true
     */
    fun getOfferRetargetingsByIds(
        shard: Int,
        clientId: ClientId,
        offerRetargetingIds: Collection<Long>
    ): Map<Long, OfferRetargeting> {
        val showConditionSelectionCriteria = ShowConditionSelectionCriteria().withShowConditionIds(offerRetargetingIds)
        val offerRetargetings =
            getOfferRetargetings(shard, clientId, showConditionSelectionCriteria, LimitOffset.maxLimited(), false)
        return offerRetargetings.associateBy { it.id }
    }


    /**
     * @return возвращает офферные ретаргетинги клиента, принадлежащие переданным группам,
     * игнорируя офферные ретаргетинги с флагом isDeleted = true
     */
    fun getOfferRetargetingsByAdGroupIds(
        shard: Int,
        clientId: ClientId,
        adGroupIds: Collection<Long>
    ): Map<Long, OfferRetargeting> {
        return getOfferRetargetingsByAdGroupIds(shard, clientId, adGroupIds, false)
    }

    /**
     * @return возвращает офферные ретаргетинги клиента, принадлежащие переданным группам
     */
    fun getOfferRetargetingsByAdGroupIds(
        shard: Int,
        clientId: ClientId,
        adGroupIds: Collection<Long>,
        withDeleted: Boolean
    ): Map<Long, OfferRetargeting> {
        val showConditionSelectionCriteria = ShowConditionSelectionCriteria().withAdGroupIds(adGroupIds)
        val offerRetargetings: List<OfferRetargeting> = getOfferRetargetings(
            shard, clientId, showConditionSelectionCriteria, LimitOffset.maxLimited(), withDeleted
        )
        return offerRetargetings.associateBy { it.adGroupId }
    }


    /**
     * @return возвращает ids офферных ретаргетингов клиента, принадлежащие переданным группам,
     * игнорируя офферные ретаргетинги с флагом isDeleted = true
     */
    fun getOfferRetargetingIdsByAdGroupIds(
        shard: Int,
        clientId: ClientId,
        adGroupIds: Collection<Long>
    ): Map<Long, List<Long>> {
        val showConditionSelectionCriteria = ShowConditionSelectionCriteria().withAdGroupIds(adGroupIds)
        val offerRetargetings = getOfferRetargetings(
            shard,
            clientId,
            showConditionSelectionCriteria,
            LimitOffset.maxLimited(),
            listOf(BidsBase.BIDS_BASE.BID_ID, BidsBase.BIDS_BASE.PID),
            false
        )
        val offerRetargetingIdsByAdGroupIds = mutableMapOf<Long, MutableList<Long>>()
        offerRetargetings.forEach { offerRetargeting ->
            offerRetargetingIdsByAdGroupIds.compute(offerRetargeting.adGroupId) { _, oldValue ->
                oldValue?.apply { add(offerRetargeting.id) } ?: mutableListOf(offerRetargeting.id)
            }
        }
        return offerRetargetingIdsByAdGroupIds
    }

    fun getOfferRetargetings(
        shard: Int,
        clientId: ClientId,
        selection: ShowConditionSelectionCriteria,
        limitOffset: LimitOffset,
        withDeleted: Boolean
    ): List<OfferRetargeting> {
        return getOfferRetargetings(shard, clientId, selection, limitOffset, jooqMapper.fieldsToRead, withDeleted)
    }

    /**
     * Возвращает офферный ретаргетинг по переданному `selection`
     */
    private fun getOfferRetargetings(
        shard: Int,
        clientId: ClientId,
        selection: ShowConditionSelectionCriteria,
        limitOffset: LimitOffset,
        fields: Collection<Field<*>>,
        withDeleted: Boolean
    ): List<OfferRetargeting> {
        if (selection.showConditionIds.isEmpty() && selection.adGroupIds.isEmpty() && selection.campaignIds.isEmpty()) {
            return emptyList()
        }
        val servingStatuses = selection.servingStatuses
        var condition = (DSL.condition(selection.campaignIds.isEmpty())
            .or(BidsBase.BIDS_BASE.CID.`in`(selection.campaignIds)))
            .and(
                DSL.condition(selection.showConditionIds.isEmpty())
                    .or(BidsBase.BIDS_BASE.BID_ID.`in`(selection.showConditionIds))
            )
            .and(
                DSL.condition(selection.adGroupIds.isEmpty())
                    .or(BidsBase.BIDS_BASE.PID.`in`(selection.adGroupIds))
            )
            .and(BidsBase.BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
            .and(
                DSL.condition(servingStatuses.isEmpty())
                    .or(Phrases.PHRASES.IS_BS_RARELY_LOADED.`in`(servingStatuses))
            )
            .and(Campaigns.CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
            .and(Campaigns.CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
            .and(Campaigns.CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
        if (!withDeleted) {
            condition = condition.andNot(
                SqlUtils.setField(BidsBase.BIDS_BASE.OPTS).contains(BidBaseOpt.DELETED.typedValue)
            )
        }
        val select = dslContextProvider.ppc(shard)
            .select(fields)
            .from(BidsBase.BIDS_BASE)
            .join(Phrases.PHRASES).on(Phrases.PHRASES.PID.eq(BidsBase.BIDS_BASE.PID))
            .join(Campaigns.CAMPAIGNS).on(Campaigns.CAMPAIGNS.CID.eq(BidsBase.BIDS_BASE.CID))
            .where(condition)
            .limit(limitOffset.limit())
            .offset(limitOffset.offset())
        return select.fetch().map(jooqMapper::fromDb)
    }

    /**
     * обновляет офферный ретаргетинг
     */
    fun update(config: Configuration, appliedChanges: List<AppliedChanges<OfferRetargeting>>) {
        updateBidsBase(config, appliedChanges)
    }

    private fun updateBidsBase(config: Configuration, appliedChanges: List<AppliedChanges<OfferRetargeting>>) {
        for (chunk in StreamEx.ofSubLists(appliedChanges, UPDATE_CHUNK_SIZE)) {
            val ub = JooqUpdateBuilder(Tables.BIDS_BASE.BID_ID, chunk)
            ub.processProperty(OfferRetargeting.PRICE, Tables.BIDS_BASE.PRICE, ::priceToDbFormat)
            ub.processProperty(OfferRetargeting.PRICE_CONTEXT, Tables.BIDS_BASE.PRICE_CONTEXT, ::priceToDbFormat)
            ub.processProperty(OfferRetargeting.LAST_CHANGE_TIME, Tables.BIDS_BASE.LAST_CHANGE)
            ub.processProperty(OfferRetargeting.STATUS_BS_SYNCED, Tables.BIDS_BASE.STATUS_BS_SYNCED) { status ->
                BidsBaseStatusbssynced.valueOf(status.toDbFormat())
            }
            ub.processProperties(
                OfferRetargetingMapping.BID_OPTION_FLAGS,
                BidsBase.BIDS_BASE.OPTS
            ) { offerRetargeting ->
                offerRetargetingPropsToDbOpts(
                    PropertyValues(
                        mutableSetOf<ModelProperty<in OfferRetargeting, *>>(
                            OfferRetargeting.IS_SUSPENDED,
                            OfferRetargeting.IS_DELETED
                        ), offerRetargeting
                    )
                )
            }
            DSL.using(config).update(Tables.BIDS_BASE)
                .set(ub.values)
                .where(Tables.BIDS_BASE.BID_ID.`in`(ub.changedIds))
                .execute()
        }
    }


    fun getOfferRetargetingIds(shard: Int, clientId: ClientId, ids: Collection<Long>): Set<Long> {
        return dslContextProvider.ppc(shard)
            .select(BidsBase.BIDS_BASE.BID_ID)
            .from(BidsBase.BIDS_BASE)
            .join(Campaigns.CAMPAIGNS).on(Campaigns.CAMPAIGNS.CID.eq(BidsBase.BIDS_BASE.CID))
            .where(Campaigns.CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
            .and(BidsBase.BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
            .and(BidsBase.BIDS_BASE.BID_ID.`in`(ids))
            .fetchSet(BidsBase.BIDS_BASE.BID_ID)
    }
}

private const val UPDATE_CHUNK_SIZE = 500
