package ru.yandex.direct.core.entity.offerretargeting.service

import org.jooq.Configuration
import org.jooq.TransactionalRunnable
import ru.yandex.direct.common.log.container.LogPriceData
import ru.yandex.direct.common.log.service.LogPriceService
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.offerretargeting.container.AdGroupInfoForOfferRetargetingAdd
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainer
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainerWithExistentAdGroups
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainerWithNonExistentAdGroups
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification.OFFER_RETARGETING_ADD
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModificationResult
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModificationResult.OFFER_RETARGETING_ADD_RESULT
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingRepository
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingDefects
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingValidationService
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.ModelProperty
import ru.yandex.direct.operation.AddedModelId
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.operation.add.ModelsValidatedStep
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.PathHelper.index
import ru.yandex.direct.validation.result.ValidationResult
import java.math.BigDecimal
import java.time.LocalDateTime

/**
 * Операция добавления офферного ретаргетинга.
 */
class OfferRetargetingAddOperation(
    applicability: Applicability,
    offerRetargetings: List<OfferRetargeting>,
    private val shard: Int,
    private val offerRetargetingRepository: OfferRetargetingRepository,
    private val offerRetargetingValidationService: OfferRetargetingValidationService,
    private val logPriceService: LogPriceService,
    private val adGroupRepository: AdGroupRepository,
    private val dslContextProvider: DslContextProvider,
    private val container: OfferRetargetingAddContainer,
) : SimpleAbstractAddOperation<OfferRetargeting, AddedModelId>(applicability, offerRetargetings),
    OfferRetargetingOperation<OfferRetargeting, AddedModelId> {
    private var additionalTask: Runnable? = null
    private var transactionalAdditionalTask: TransactionalRunnable? = null

    fun setAdGroupInfos(adGroupInfos: Map<Int, AdGroupInfoForOfferRetargetingAdd>) {
        check(container is OfferRetargetingAddContainerWithNonExistentAdGroups) { "ad group info can't be set if groups exist" }
        check(!isPrepared) { "operation is already prepared" }
        container.adGroupInfos = adGroupInfos
    }

    fun setAdGroupsIds(adGroupIds: Map<Int, Long>) {
        check(container is OfferRetargetingAddContainerWithNonExistentAdGroups) {
            "ids can be set only when ad groups non existent on prepare"
        }
        check(isPrepared) { "operation is not prepared yet" }
        check(!isExecuted) { "operation is already executed" }
        check(models.indices.all { it in adGroupIds }) {
            val indicesWithoutAdGroupId = models.indices.filter { it !in adGroupIds }
            "adGroupIds haven't set for indices: $indicesWithoutAdGroupId"
        }
        for (i in models.indices) {
            models[i].adGroupId = adGroupIds[i]
        }
    }

    override fun validate(preValidationResult: ValidationResult<List<OfferRetargeting>, Defect<*>>) {
        when (container) {
            is OfferRetargetingAddContainerWithExistentAdGroups ->
                offerRetargetingValidationService.validateAddOfferRetargetings(preValidationResult, container)
            is OfferRetargetingAddContainerWithNonExistentAdGroups -> {
                // проверяем, что информация о добавляемых группах выставлена на этапе валидации
                container.checkConsistency(preValidationResult.value.size)
                offerRetargetingValidationService.validateAddOfferRetargetings(preValidationResult, container)
            }
        }
    }

    override fun onModelsValidated(modelsValidatedStep: ModelsValidatedStep<OfferRetargeting>) {
        val offerRetargetings = modelsValidatedStep.validModelsMap
        prepareSystemFields(offerRetargetings)
    }

    private fun prepareSystemFields(offerRetargetings: Map<Int, OfferRetargeting>) {
        val now = LocalDateTime.now()
        offerRetargetings.forEach { (index, offerRetargeting) ->
            val campaignId = container.getCampaign(index, offerRetargeting.adGroupId)?.id
            offerRetargeting
                .withLastChangeTime(now)
                .withStatusBsSynced(StatusBsSynced.NO)
                .withCampaignId(campaignId)
        }
    }

    override fun beforeExecution(validModelsMapToApply: Map<Int, OfferRetargeting>) {
        val offerRetargetings = validModelsMapToApply.values
        setMissingPrices(offerRetargetings)

        computeAdditionalTask(offerRetargetings)
        computeTransactionalTask(offerRetargetings)
    }

    /**
     * Выставление цен по умолчанию, если они явно не указаны
     *
     * @param offerRetargetings офферные ретаргетинги, которым возможно нужно выставить цены
     */
    private fun setMissingPrices(offerRetargetings: Collection<OfferRetargeting>) {
        offerRetargetings.forEach {
            it.price = it.price ?: BigDecimal.ZERO
            it.priceContext = it.priceContext ?: BigDecimal.ZERO
        }
    }

    private fun computeAdditionalTask(offerRetargetings: Collection<OfferRetargeting>) {
        additionalTask = Runnable {
            val priceDataList = computeLogPriceDataList(offerRetargetings)
            logPriceService.logPrice(priceDataList, container.operatorUid)
        }
    }

    private fun computeLogPriceDataList(newOfferRetargeting: Collection<OfferRetargeting>): List<LogPriceData> {
        return newOfferRetargeting.map { offerRetargeting: OfferRetargeting ->
            requireNotNull(offerRetargeting.id) {
                buildString {
                    append("attempt to log offerRetargeting without id (maybe repository does not set id to added")
                    append("offerRetargeting or computing log records is called before offerRetargeting is added).")
                }
            }
            val priceSearch: Double = offerRetargeting.price?.toDouble() ?: 0.0
            val priceContext: Double = offerRetargeting.priceContext?.toDouble() ?: 0.0
            LogPriceData(
                offerRetargeting.campaignId,
                offerRetargeting.adGroupId,
                offerRetargeting.id,
                priceContext,
                priceSearch,
                container.clientCurrency.code,
                LogPriceData.OperationType.INSERT_1
            )
        }
    }

    private fun computeTransactionalTask(offerRetargetings: Collection<OfferRetargeting>) {
        val affectedAdGroupIds: Set<Long> = offerRetargetings.mapTo(mutableSetOf()) { it.adGroupId }
        transactionalAdditionalTask = TransactionalRunnable { conf: Configuration ->
            if (affectedAdGroupIds.isNotEmpty()) {
                adGroupRepository.dropStatusModerateExceptDraftsAndModerated(conf, affectedAdGroupIds)
                adGroupRepository.updateStatusBsSyncedExceptNew(conf, affectedAdGroupIds, StatusBsSynced.NO)
                adGroupRepository.updateLastChange(conf, affectedAdGroupIds)
            }
        }
    }

    override fun execute(validModelsToApply: List<OfferRetargeting>): List<AddedModelId> {
        val offerRetargetingIds: MutableList<AddedModelId> = ArrayList()
        val saveFn = TransactionalRunnable { conf: Configuration ->
            val affectedAdGroupIds: Set<Long> = validModelsToApply.mapTo(mutableSetOf()) { it.adGroupId }
            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds)
            offerRetargetingIds.addAll(
                offerRetargetingRepository.addOfferRetargetings(
                    conf,
                    container.clientId,
                    validModelsToApply,
                    affectedAdGroupIds
                )
            )
            transactionalAdditionalTask!!.run(conf)
        }
        Trace.current().profile("offerRetargetingAdd:write", "", validModelsToApply.size.toLong()).use {
            dslContextProvider.ppcTransaction(shard, saveFn)
            additionalTask!!.run()
        }
        return offerRetargetingIds
    }

    override fun createMassResult(
        resultMap: Map<Int, AddedModelId>,
        validationResult: ValidationResult<List<OfferRetargeting>, Defect<*>>,
        canceledElementIndexes: Set<Int>
    ): MassResult<AddedModelId> {
        for (model in modifyingModels.withIndex()) {
            val modelId = resultMap[model.index]
            if (modelId != null) {
                val subValidationResult =
                    validationResult.getOrCreateSubValidationResult(index(model.index), model.value)

                if (!modelId.isAdded) {
                    subValidationResult.addWarning(OfferRetargetingDefects.tooManyOfferRetargetingsInAdGroup())
                }
            }
        }
        return super.createMassResult(resultMap, validationResult, canceledElementIndexes)
    }

    override val modificationModelProperty: ModelProperty<OfferRetargetingModification, List<OfferRetargeting>>
        get() = OFFER_RETARGETING_ADD

    override val modifyingModels: List<OfferRetargeting> = offerRetargetings

    override val modificationResultModelProperty: ModelProperty<OfferRetargetingModificationResult, List<Long>>
        get() = OFFER_RETARGETING_ADD_RESULT

    override val resultOfModification: List<Long>
        get() = result.get().result.map { it.result.id }
}

