package ru.yandex.direct.core.entity.offerretargeting.service

import org.jooq.Configuration
import org.jooq.TransactionalRunnable
import ru.yandex.direct.common.log.container.LogPriceData
import ru.yandex.direct.common.log.service.LogPriceService
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification.OFFER_RETARGETING_IDS_DELETE
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModificationResult
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModificationResult.OFFER_RETARGETING_DELETE_RESULT
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingRepository
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingValidationService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.model.ModelProperty
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

class OfferRetargetingDeleteOperation(
    applicability: Applicability,
    modelIds: List<Long>,
    private val shard: Int,
    private val clientId: ClientId,
    private val operatorUid: Long,
    private val offerRetargetingRepository: OfferRetargetingRepository,
    private val offerRetargetingValidationService: OfferRetargetingValidationService,
    private val logPriceService: LogPriceService,
    private val adGroupRepository: AdGroupRepository,
    private val dslContextProvider: DslContextProvider,
    // офферные ретаргетинги по id, которые собираемся удалить
    private val offerRetargetingsByIds: Map<Long, OfferRetargeting>,
) : AbstractOperationWithId(applicability, modelIds), OfferRetargetingOperation<Long, Long> {
    private var additionalTask: Runnable? = null
    private var transactionalAdditionalTask: TransactionalRunnable? = null
    private var appliedChanges: List<AppliedChanges<OfferRetargeting>>? = null


    override fun validate(ids: List<Long>): ValidationResult<List<Long>, Defect<*>> =
        offerRetargetingValidationService.validateDeleteOfferRetargetings(
            ids, offerRetargetingsByIds, operatorUid, clientId
        )

    override fun beforeExecution(ids: List<Long>) {
        computeAdditionalTask(ids)
        computeTransactionalTask(ids)

        // удаление офферного ретаргетинга заключается в обнулении полей
        appliedChanges = ids.map { id ->
            val offerRetargetingModelChanges = prepareToDelete(id)
            offerRetargetingModelChanges.applyTo(offerRetargetingsByIds.getValue(offerRetargetingModelChanges.id))
        }
    }

    private fun computeAdditionalTask(ids: List<Long>) {
        val offerRetargetings = ids.map { offerRetargetingsByIds.getValue(it) }

        val priceDataList = computeLogPriceDataList(offerRetargetings)
        additionalTask = Runnable { logPriceService.logPrice(priceDataList, operatorUid) }
    }

    private fun computeLogPriceDataList(newOfferRetargeting: Collection<OfferRetargeting>): List<LogPriceData> =
        newOfferRetargeting.map(::offerRetargetingToLog)

    private fun offerRetargetingToLog(offerRetargeting: OfferRetargeting): LogPriceData =
        LogPriceData(
            offerRetargeting.campaignId,
            offerRetargeting.adGroupId,
            offerRetargeting.id,
            null,
            null,
            null,
            LogPriceData.OperationType.DELETE_1
        )

    private fun computeTransactionalTask(ids: List<Long>) {
        val affectedAdGroupIds = ids.mapTo(mutableSetOf()) {
            val offerRetargeting = offerRetargetingsByIds.getValue(it)
            offerRetargeting.adGroupId
        } as Set<Long>

        transactionalAdditionalTask = TransactionalRunnable { conf: Configuration ->
            if (affectedAdGroupIds.isNotEmpty()) {
                adGroupRepository.updateStatusBsSyncedExceptNew(conf, affectedAdGroupIds, StatusBsSynced.NO)
                adGroupRepository.updateLastChange(conf, affectedAdGroupIds)
            }
        }
    }

    private fun prepareToDelete(id: Long): ModelChanges<OfferRetargeting> {
        val changes = ModelChanges(id, OfferRetargeting::class.java)
        changes.process(null, OfferRetargeting.PRICE)
        changes.process(null, OfferRetargeting.PRICE_CONTEXT)
        changes.process(true, OfferRetargeting.IS_DELETED)
        return changes
    }

    override fun execute(ids: List<Long>) {
        val saveFn = TransactionalRunnable { conf: Configuration ->
            val affectedAdGroupIds =
                offerRetargetingsByIds.values.asSequence()
                    .filter { ids.contains(it.id) }
                    .map { it.adGroupId }
                    .toSet()
            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds)
            offerRetargetingRepository.update(conf, appliedChanges!!)
            transactionalAdditionalTask!!.run(conf)
        }

        dslContextProvider.ppcTransaction(shard, saveFn)
    }

    override fun afterExecution(ids: List<Long>) {
        additionalTask!!.run()
    }

    override val modificationModelProperty: ModelProperty<OfferRetargetingModification, List<Long>>
        get() = OFFER_RETARGETING_IDS_DELETE

    override val modifyingModels: List<Long> = modelIds

    override val modificationResultModelProperty: ModelProperty<OfferRetargetingModificationResult, List<Long>>
        get() = OFFER_RETARGETING_DELETE_RESULT

    override val resultOfModification: List<Long>
        get() = result.get().result.map { it.result }

}
