package ru.yandex.direct.core.entity.offerretargeting.service

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainerWithExistentAdGroups
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModificationResult
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingUpdateContainer
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.operation.Operation
import ru.yandex.direct.result.Result
import ru.yandex.direct.result.Result.broken
import ru.yandex.direct.result.Result.successful
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.PathHelper.field
import ru.yandex.direct.validation.result.ValidationResult

/**
 * Комплексная операция обновления/добавления/удаления офферных ретаргетингов
 */
class OfferRetargetingModifyOperation(
    offerRetargetingService: OfferRetargetingService,
    clientService: ClientService,
    clientId: ClientId,
    clientUid: Long,
    operatorUid: Long,
    shard: Int,
    private val offerRetargetingModification: OfferRetargetingModification,
    campaignRepository: CampaignRepository,
    adGroupRepository: AdGroupRepository,
    offerRetargetingRepository: OfferRetargetingRepository,
) {
    private val operations: List<OfferRetargetingOperation<*, *>>
    private var result: Result<OfferRetargetingModificationResult>? = null
    private var prepared = false
    private var executed = false

    init {
        val updatingOfferRetargetingIds: Set<Long> =
            offerRetargetingModification.offerRetargetingUpdate.mapTo(mutableSetOf()) { it.id }
        val offerRetargetingIds: Set<Long> =
            updatingOfferRetargetingIds + offerRetargetingModification.offerRetargetingIdsDelete

        val offerRetargetingsByIds = offerRetargetingRepository.getOfferRetargetingsByIds(shard, clientId, offerRetargetingIds)

        val operations = mutableListOf<OfferRetargetingOperation<*, *>>()

        // инициализируем операцию удаления
        if (offerRetargetingModification.offerRetargetingIdsDelete.isNotEmpty()) {
            val deletingOfferRetargetingsByIds = offerRetargetingsByIds.filterKeys {
                it in offerRetargetingModification.offerRetargetingIdsDelete
            }
            operations += offerRetargetingService.createFullDeleteOperation(
                clientId,
                operatorUid,
                offerRetargetingModification.offerRetargetingIdsDelete,
                deletingOfferRetargetingsByIds
            )
        }

        val updatingOfferRetargetingsByIds = offerRetargetingsByIds.filterKeys { it in updatingOfferRetargetingIds }
        val currency = clientService.getWorkCurrency(clientId)

        val adGroupIds: Set<Long> =
            (updatingOfferRetargetingsByIds.values.asSequence() + offerRetargetingModification.offerRetargetingAdd.asSequence())
                .mapTo(mutableSetOf()) { it.adGroupId }

        val clientCampaignIdsByAdGroupIds = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds)
            .mapValues { (_, adGroup) -> adGroup.campaignId }
        val clientCampaignIds = clientCampaignIdsByAdGroupIds.values.toList()
        val clientCampaignsByIds = campaignRepository.getCampaigns(shard, clientCampaignIds).associateBy { it.id }

        // инициализируем операцию добавления
        if (offerRetargetingModification.offerRetargetingAdd.isNotEmpty()) {
            val addOperationContainer = OfferRetargetingAddContainerWithExistentAdGroups(
                operatorUid,
                clientId,
                currency,
                clientCampaignsByIds,
                clientCampaignIdsByAdGroupIds
            )
            operations += offerRetargetingService.createFullAddOperation(
                offerRetargetingModification.offerRetargetingAdd,
                addOperationContainer
            )
        }

        val adGroupIdsByOfferRetargetingIds = updatingOfferRetargetingsByIds.mapValues { (_, value) -> value.adGroupId }
        // инициализируем операцию обновления
        if (offerRetargetingModification.offerRetargetingUpdate.isNotEmpty()) {
            val updateOperationContainer = OfferRetargetingUpdateContainer(
                operatorUid,
                clientId,
                clientUid,
                currency,
                clientCampaignsByIds,
                clientCampaignIdsByAdGroupIds,
                adGroupIdsByOfferRetargetingIds,
                updatingOfferRetargetingsByIds
            )
            operations += offerRetargetingService.createFullUpdateOperation(
                offerRetargetingModification.offerRetargetingUpdate,
                updateOperationContainer
            )
        }
        this.operations = operations
    }

    fun prepare(): Result<OfferRetargetingModificationResult>? {
        check(!prepared) { "prepare() can be called only once" }
        prepared = true
        val failed = operations.fold(false) { acc, operation: Operation<*> -> operation.prepare().isPresent || acc }
        if (failed) {
            val validationResult = buildValidationResult()
            result = broken(validationResult)
        }
        return result
    }

    fun prepareAndApply(): Result<OfferRetargetingModificationResult> {
        return prepare() ?: apply()
    }

    private fun buildValidationResult(): ValidationResult<OfferRetargetingModification, Defect<*>> {
        val validationResult = ValidationResult<OfferRetargetingModification, Defect<*>>(offerRetargetingModification)
        for (operation in operations) {
            if (operation.result.isPresent) {
                val operationResult = validationResult.getOrCreateSubValidationResult(
                    field(operation.modificationModelProperty.name()),
                    operation.modifyingModels
                )
                val vr = operation.result.get().validationResult
                if (vr != null) {
                    ValidationResult.transferSubNodesWithIssues(vr, operationResult)
                }
            }
        }
        return validationResult
    }

    fun apply(): Result<OfferRetargetingModificationResult> {
        check(prepared) { "prepare() must be called before apply()" }
        check(!executed) { "apply() can be called only once" }
        check(result == null) { "result is already computed by prepare()" }

        executed = true
        val offerRetargetingModificationResult = OfferRetargetingModificationResult()
        for (operation in operations) {
            operation.apply()
            operation.modificationResultModelProperty.set(
                offerRetargetingModificationResult,
                operation.resultOfModification
            )
        }
        val validationResult = buildValidationResult()
        return successful(offerRetargetingModificationResult, validationResult).also { result = it }
    }
}
