package ru.yandex.direct.core.entity.offerretargeting.service

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.common.log.service.LogPriceService
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.EntityService
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService
import ru.yandex.direct.core.entity.offerretargeting.container.AdGroupInfoForOfferRetargetingAdd
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainer
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainerWithExistentAdGroups
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainerWithNonExistentAdGroups
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingUpdateContainer
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingRepository
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingValidationService
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.operation.ConvertResultOperation
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.result.ResultConverters.massResultValueConverter
import ru.yandex.direct.utils.converter.Converters.nullSafeConverter

@Service
class OfferRetargetingService @Autowired constructor(
    private val shardHelper: ShardHelper,
    private val offerRetargetingValidationService: OfferRetargetingValidationService,
    private val offerRetargetingRepository: OfferRetargetingRepository,
    private val adGroupRepository: AdGroupRepository,
    private val logPriceService: LogPriceService,
    private val dslContextProvider: DslContextProvider,
    private val clientService: ClientService,
    private val campaignRepository: CampaignRepository,
    private val mailNotificationEventService: MailNotificationEventService,
    private val aggregatedStatusesRepository: AggregatedStatusesRepository
) : EntityService<OfferRetargeting, Long> {
    fun createFullDeleteOperation(
        clientId: ClientId,
        operatorUid: Long,
        ids: List<Long>,
        offerRetargetingsByIds: Map<Long, OfferRetargeting>
    ): OfferRetargetingDeleteOperation {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        return createDeleteOperation(shard, operatorUid, clientId, ids, Applicability.FULL, offerRetargetingsByIds)
    }

    fun createDeleteOperation(
        clientId: ClientId,
        operatorUid: Long,
        ids: List<Long>,
        applicability: Applicability
    ): OfferRetargetingDeleteOperation {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val offerRetargetingsByIds = offerRetargetingRepository.getOfferRetargetingsByIds(shard, clientId, ids)
        return createDeleteOperation(shard, operatorUid, clientId, ids, applicability, offerRetargetingsByIds)
    }

    private fun createDeleteOperation(
        shard: Int,
        operatorUid: Long,
        clientId: ClientId,
        ids: List<Long>,
        applicability: Applicability,
        offerRetargetingsByIds: Map<Long, OfferRetargeting>
    ): OfferRetargetingDeleteOperation {
        return OfferRetargetingDeleteOperation(
            applicability,
            ids,
            shard,
            clientId,
            operatorUid,
            offerRetargetingRepository,
            offerRetargetingValidationService,
            logPriceService,
            adGroupRepository,
            dslContextProvider,
            offerRetargetingsByIds
        )
    }

    /**
     * Получить множество идентификаторов, соотвествующих OfferRetargeting клиента, по списку
     * предположительных идентификаторов
     */
    fun getOfferRetargetingIds(clientId: ClientId, ids: Collection<Long>): Set<Long> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        return offerRetargetingRepository.getOfferRetargetingIds(shard, clientId, ids)
    }

    override operator fun get(clientId: ClientId, operatorUid: Long, ids: Collection<Long>): List<OfferRetargeting> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        return offerRetargetingRepository.getOfferRetargetingsByIds(shard, clientId, ids).values.toList()
    }

    fun createFullAddOperation(
        offerRetargetings: List<OfferRetargeting>,
        container: OfferRetargetingAddContainer,
    ): OfferRetargetingAddOperation {
        val shard = shardHelper.getShardByClientIdStrictly(container.clientId)
        return createAddOperation(Applicability.FULL, offerRetargetings, shard, container)
    }

    fun createFullAddOperation(
        offerRetargetings: List<OfferRetargeting>,
        clientId: ClientId,
        operatorUid: Long,
        clientCurrency: Currency
    ): OfferRetargetingAddOperation {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val addOperationContainer: OfferRetargetingAddContainer =
            createAddContainer(offerRetargetings, operatorUid, clientId, shard, clientCurrency)

        return createAddOperation(Applicability.FULL, offerRetargetings, shard, addOperationContainer)
    }

    /**
     * Создает операцию добавления офферных ретаргетингов, которая будет выполняться в рамках операции создания групп
     *
     * @param adGroupInfos информация о создаваемых группах по индексу офферного ретаргетинга
     * @see OfferRetargetingAddContainerWithNonExistentAdGroups
     */
    fun createAddOperationWithNonExistingAdGroups(
        offerRetargetings: List<OfferRetargeting>,
        clientId: ClientId,
        operatorUid: Long,
        clientCurrency: Currency,
        adGroupInfos: Map<Int, AdGroupInfoForOfferRetargetingAdd> = emptyMap()
    ): OfferRetargetingAddOperation {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val addOperationContainer =
            OfferRetargetingAddContainerWithNonExistentAdGroups(operatorUid, clientId, clientCurrency, adGroupInfos)
        return createAddOperation(
            Applicability.FULL,
            offerRetargetings,
            shard,
            addOperationContainer,
        )
    }

    private fun createAddOperation(
        applicability: Applicability,
        offerRetargetings: List<OfferRetargeting>,
        shard: Int,
        addOperationContainer: OfferRetargetingAddContainer,
    ): OfferRetargetingAddOperation {
        return OfferRetargetingAddOperation(
            applicability,
            offerRetargetings,
            shard,
            offerRetargetingRepository,
            offerRetargetingValidationService,
            logPriceService,
            adGroupRepository,
            dslContextProvider,
            addOperationContainer,
        )
    }

    fun createAddContainer(
        offerRetargetingsToAdd: List<OfferRetargeting>,
        operatorUid: Long,
        clientId: ClientId,
        shard: Int,
        clientCurrency: Currency
    ): OfferRetargetingAddContainer {
        // достаем id кампаний через группы создаваемых офферных ретаргетингов
        val adGroupIds: Set<Long> = offerRetargetingsToAdd.mapTo(mutableSetOf()) { it.adGroupId }
        val clientCampaignIdsByAdGroupIds = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds)
            .mapValues { (_, adGroup) -> adGroup.campaignId }

        // достаем кампании по id
        val clientCampaignsByIds =
            campaignRepository.getClientsCampaignsByIds(shard, clientId, clientCampaignIdsByAdGroupIds.values)
                .associateBy { it.id }

        return OfferRetargetingAddContainerWithExistentAdGroups(
            operatorUid,
            clientId,
            clientCurrency,
            clientCampaignsByIds,
            clientCampaignIdsByAdGroupIds
        )
    }

    fun createFullUpdateOperation(
        offerRetargetingModelChangesList: List<ModelChanges<OfferRetargeting>>,
        container: OfferRetargetingUpdateContainer
    ): OfferRetargetingUpdateOperation {
        val shard = shardHelper.getShardByClientIdStrictly(container.clientId)
        return createUpdateOperation(
            Applicability.FULL,
            offerRetargetingModelChangesList,
            container,
            shard
        )
    }

    fun createFullUpdateOperation(
        offerRetargetingModelChangesList: List<ModelChanges<OfferRetargeting>>,
        clientId: ClientId,
        clientUid: Long,
        clientCurrency: Currency,
        operatorUid: Long
    ): OfferRetargetingUpdateOperation {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val offerRetargetingUpdateOperationContainer = createUpdateContainer(
            offerRetargetingModelChangesList,
            operatorUid,
            clientId,
            clientUid,
            clientCurrency,
            shard
        )
        return createUpdateOperation(
            Applicability.FULL,
            offerRetargetingModelChangesList,
            offerRetargetingUpdateOperationContainer,
            shard,
        )
    }

    fun createPartialUpdateOperation(
        offerRetargetingModelChangesList: List<ModelChanges<OfferRetargeting>>,
        clientId: ClientId,
        clientUid: Long,
        clientCurrency: Currency,
        operatorUid: Long
    ): OfferRetargetingUpdateOperation {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val offerRetargetingUpdateOperationContainer = createUpdateContainer(
            offerRetargetingModelChangesList,
            operatorUid,
            clientId,
            clientUid,
            clientCurrency,
            shard
        )
        return createUpdateOperation(
            Applicability.PARTIAL,
            offerRetargetingModelChangesList,
            offerRetargetingUpdateOperationContainer,
            shard
        )
    }

    private fun createUpdateOperation(
        applicability: Applicability,
        offerRetargetingModelChangesList: List<ModelChanges<OfferRetargeting>>,
        container: OfferRetargetingUpdateContainer,
        shard: Int,
    ): OfferRetargetingUpdateOperation {
        return OfferRetargetingUpdateOperation(
            applicability,
            offerRetargetingModelChangesList,
            logPriceService,
            mailNotificationEventService,
            offerRetargetingValidationService,
            offerRetargetingRepository,
            adGroupRepository,
            aggregatedStatusesRepository,
            dslContextProvider,
            container,
            shard,
        )
    }

    private fun createUpdateContainer(
        offerRetargetingsForUpdate: List<ModelChanges<OfferRetargeting>>,
        operatorUid: Long,
        clientId: ClientId,
        clientUid: Long,
        clientCurrency: Currency,
        shard: Int
    ): OfferRetargetingUpdateContainer {
        // достаем офферные ретаргетинги по id
        val offerRetargetingIds = offerRetargetingsForUpdate.map { it.id }
        val offerRetargetingsByIds =
            offerRetargetingRepository.getOfferRetargetingsByIds(shard, clientId, offerRetargetingIds)

        // достаем id групп офферных ретаргетингов, для которых применяются изменения
        val adGroupIdsByOfferRetargetingIds = offerRetargetingsByIds.mapValues { (_, value) -> value.adGroupId }

        // достаем id кампаний через группы, для которых примененяются изменения
        val adGroupIds: Set<Long> = offerRetargetingsByIds.values.mapTo(mutableSetOf()) { it.adGroupId }
        val clientCampaignIdsByAdGroupIds = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds)
            .mapValues { (_, adGroup) -> adGroup.campaignId }

        // достаем кампании по id
        val clientCampaignsByIds =
            campaignRepository.getClientsCampaignsByIds(shard, clientId, clientCampaignIdsByAdGroupIds.values)
                .associateBy { it.id }

        return OfferRetargetingUpdateContainer(
            operatorUid,
            clientId,
            clientUid,
            clientCurrency,
            clientCampaignsByIds,
            clientCampaignIdsByAdGroupIds,
            adGroupIdsByOfferRetargetingIds,
            offerRetargetingsByIds
        )
    }

    fun createFullModifyOperation(
        clientId: ClientId,
        clientUid: Long,
        operatorUid: Long,
        offerRetargetingModification: OfferRetargetingModification,
    ): OfferRetargetingModifyOperation {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        return OfferRetargetingModifyOperation(
            this,
            clientService,
            clientId,
            clientUid,
            operatorUid,
            shard,
            offerRetargetingModification,
            campaignRepository,
            adGroupRepository,
            offerRetargetingRepository
        )
    }

    override fun add(
        clientId: ClientId,
        operatorUid: Long,
        entities: List<OfferRetargeting>,
        applicability: Applicability
    ): MassResult<Long?> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val client = requireNotNull(clientService.getClient(clientId))

        val container = createAddContainer(entities, operatorUid, clientId, shard, client.workCurrency.currency)

        val operation = ConvertResultOperation(
            createAddOperation(
                applicability,
                entities,
                shard,
                container
            ),
            massResultValueConverter(nullSafeConverter { it.id })
        )
        return operation.prepareAndApply()
    }

    override fun copy(
        copyContainer: CopyOperationContainer,
        entities: List<OfferRetargeting>,
        applicability: Applicability
    ): MassResult<Long?> {
        val shard = copyContainer.shardTo
        val client = copyContainer.clientTo

        val addContainer = createAddContainer(
            entities, copyContainer.operatorUid, copyContainer.clientIdTo, shard, client.workCurrency.currency)

        val operation = ConvertResultOperation(
            createAddOperation(
                applicability,
                entities,
                shard,
                addContainer
            ),
            massResultValueConverter(nullSafeConverter { it.id })
        )
        return operation.prepareAndApply()
    }

}
