package ru.yandex.direct.core.entity.offerretargeting.service

import com.google.common.collect.Sets
import org.jooq.Configuration
import org.jooq.TransactionalRunnable
import ru.yandex.direct.common.log.container.LogPriceData
import ru.yandex.direct.common.log.service.LogPriceService
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent
import ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent.changedSearchPriceEvent
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModification.OFFER_RETARGETING_UPDATE
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModificationResult
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingModificationResult.OFFER_RETARGETING_UPDATE_RESULT
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingUpdateContainer
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingRepository
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingValidationService
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.model.ModelProperty
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.operation.update.ExecutionStep
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.math.BigDecimal
import java.time.LocalDateTime


class OfferRetargetingUpdateOperation(
    applicability: Applicability,
    offerRetargetingModelChangesList: List<ModelChanges<OfferRetargeting>>,
    private val logPriceService: LogPriceService,
    private val mailNotificationEventService: MailNotificationEventService,
    private val offerRetargetingValidationService: OfferRetargetingValidationService,
    private val offerRetargetingRepository: OfferRetargetingRepository,
    private val adGroupRepository: AdGroupRepository,
    private val aggregatedStatusesRepository: AggregatedStatusesRepository,
    private val dslContextProvider: DslContextProvider,
    private val container: OfferRetargetingUpdateContainer,
    private val shard: Int,
) : SimpleAbstractUpdateOperation<OfferRetargeting, Long>(
    applicability,
    offerRetargetingModelChangesList,
    { OfferRetargeting().withId(it) }
), OfferRetargetingOperation<ModelChanges<OfferRetargeting>, Long> {
    private var additionalTask: Runnable? = null
    private var transactionalAdditionalTask: TransactionalRunnable? = null

    private val clientId: ClientId
        get() = container.clientId
    private val operatorUid: Long
        get() = container.operatorUid
    private val clientUid: Long
        get() = container.clientUid
    private val clientCurrency: Currency
        get() = container.clientCurrency


    override fun validateModelChanges(
        modelChanges: List<ModelChanges<OfferRetargeting>>
    ): ValidationResult<List<ModelChanges<OfferRetargeting>>, Defect<*>> =
        offerRetargetingValidationService.preValidateUpdateOfferRetargetings(modelChanges, container)

    override fun getModels(ids: Collection<Long>): List<OfferRetargeting> = ids.map {
        container.offerRetargetingsByIds[it]!!
    }

    override fun validateAppliedChanges(validationResult: ValidationResult<List<OfferRetargeting>, Defect<*>>) =
        offerRetargetingValidationService.validateUpdateOfferRetargetings(validationResult, container)

    override fun beforeExecution(executionStep: ExecutionStep<OfferRetargeting>) {
        val validAppliedChanges = executionStep.validAppliedChanges
        validAppliedChanges.forEach { appliedChanges -> prepareStatuses(appliedChanges) }
        computeTransactionalTask(validAppliedChanges)
        computeAdditionalTask(validAppliedChanges)
    }

    private fun computeAdditionalTask(validAppliedChanges: Collection<AppliedChanges<OfferRetargeting>>) {
        val offerRetargetings = validAppliedChanges.asSequence()
            .filter { it.changed(OfferRetargeting.PRICE) || it.changed(OfferRetargeting.PRICE_CONTEXT) }
            .map { it.model }
            .toList()
        val priceDataList = computeLogPriceDataList(offerRetargetings)
        val events = computeMailEvents(validAppliedChanges)

        additionalTask = Runnable {
            if (priceDataList.isNotEmpty()) {
                logPriceService.logPrice(priceDataList, operatorUid)
            }
            if (events.isNotEmpty()) {
                mailNotificationEventService.queueEvents(operatorUid, clientId, events)
            }
        }
    }

    private fun prepareStatuses(appliedChanges: AppliedChanges<OfferRetargeting>) {
        val now = LocalDateTime.now()
        if (isBsSyncStatusChanged(appliedChanges)) {
            appliedChanges.modify(OfferRetargeting.STATUS_BS_SYNCED, StatusBsSynced.NO)
        }
        if (appliedChanges.hasActuallyChangedProps()) {
            appliedChanges.modify(OfferRetargeting.LAST_CHANGE_TIME, now)
        }
    }

    private fun computeTransactionalTask(validAppliedChanges: Collection<AppliedChanges<OfferRetargeting>>) {
        val adGroupIdsToResend = validAppliedChanges.asSequence()
            .filter { isAdGroupBsSyncStatusChanged(it) }
            .map { it.model.adGroupId }
            .toSet()

        val validAppliedChangesSuspended = validAppliedChanges.asSequence()
            .filter { it.changed(OfferRetargeting.IS_SUSPENDED) }
        // Если на группе включился офферный ретаргетинг и она не промодерирована - переотправляем на модерацию
        // т.к. с офферным ретаргетингом, возможно, она будет пропущена
        val adGroupIdsToModerate = validAppliedChangesSuspended
            .map { it.model.adGroupId }
            .toList()

        val adGroupIdsToUpdateLastChange = mutableSetOf<Long>().apply {
            addAll(adGroupIdsToModerate)
            addAll(adGroupIdsToResend)
        }
        val offerRetargetingIdsToResetStatuses = validAppliedChangesSuspended
            .map { it.model.id }
            .toList()

        transactionalAdditionalTask = TransactionalRunnable { conf: Configuration ->
            if (adGroupIdsToResend.isNotEmpty()) {
                adGroupRepository.updateStatusBsSyncedExceptNew(conf, adGroupIdsToResend, StatusBsSynced.NO)
            }
            if (adGroupIdsToModerate.isNotEmpty()) {
                adGroupRepository.dropStatusModerateExceptDraftsAndModerated(conf, adGroupIdsToModerate)
            }
            adGroupRepository.updateLastChange(conf, adGroupIdsToUpdateLastChange)
            aggregatedStatusesRepository.markKeywordStatusesAsObsolete(
                conf.dsl(), null, offerRetargetingIdsToResetStatuses
            )
        }
    }

    private fun isBsSyncStatusChanged(appliedChanges: AppliedChanges<OfferRetargeting>): Boolean {
        return appliedChanges.changed(OfferRetargeting.PRICE) ||
            appliedChanges.changed(OfferRetargeting.PRICE_CONTEXT)
    }

    private fun isAdGroupBsSyncStatusChanged(appliedChanges: AppliedChanges<OfferRetargeting>): Boolean {
        return appliedChanges.changed(OfferRetargeting.IS_DELETED)
            || appliedChanges.changed(OfferRetargeting.IS_SUSPENDED)
    }

    private fun computeLogPriceDataList(newOfferRetargeting: Collection<OfferRetargeting>): List<LogPriceData> {
        return newOfferRetargeting.map {
            val priceSearch = it.price?.toDouble() ?: 0.0
            val priceContext = it.priceContext?.toDouble() ?: 0.0

            LogPriceData(
                it.campaignId,
                it.adGroupId,
                it.id,
                priceContext,
                priceSearch,
                clientCurrency.code,
                LogPriceData.OperationType.UPDATE_1
            )
        }
    }

    private fun computeMailEvents(changes: Collection<AppliedChanges<OfferRetargeting>>): List<KeywordEvent<BigDecimal>> {
        val affectedAdGroupIds: Set<Long> = changes.mapTo(mutableSetOf()) { it.model.adGroupId }
        val adGroupIdToAdGroup = getAdGroupIdToAdGroupNameWithCheck(affectedAdGroupIds)
        return changes.asSequence().filter {
            it.changed(OfferRetargeting.PRICE)
                && it.getOldValue(OfferRetargeting.PRICE) != null && it.getNewValue(OfferRetargeting.PRICE) != null
        }.map {
            val adGroupId = it.model.adGroupId
            val campaignId = it.model.campaignId
            val adGroupName = adGroupIdToAdGroup[adGroupId]!!.name

            val oldValue = checkNotNull(it.getOldValue(OfferRetargeting.PRICE)) { "Old price value can't be null" }
            val newValue = checkNotNull(it.getNewValue(OfferRetargeting.PRICE)) { "New price value can't be null" }

            changedSearchPriceEvent(operatorUid, clientUid, campaignId, adGroupId, adGroupName, oldValue, newValue)
        }.toList()
    }

    private fun getAdGroupIdToAdGroupNameWithCheck(adGroupIds: Set<Long>): Map<Long, AdGroupName> {
        val adGroupIdToAdGroup = adGroupRepository.getAdGroupNames(shard, clientId, adGroupIds)

        check(adGroupIdToAdGroup.keys.containsAll(adGroupIds)) {
            val notFoundAdGroupIds = Sets.difference(adGroupIds, adGroupIdToAdGroup.keys)
            "can't get adGroups for adGroup ids: $notFoundAdGroupIds"
        }
        return adGroupIdToAdGroup
    }

    override fun execute(applicableAppliedChanges: List<AppliedChanges<OfferRetargeting>>): List<Long> {
        Trace.current()
            .profile("offerRetargetingUpdate:write", "", applicableAppliedChanges.size.toLong()).use {
                dslContextProvider.ppcTransaction(shard) { conf: Configuration ->
                    val affectedAdGroupIds: Set<Long> = applicableAppliedChanges.mapTo(mutableSetOf()) {
                        it.model.adGroupId
                    }
                    adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds)
                    offerRetargetingRepository.update(conf, applicableAppliedChanges)
                    transactionalAdditionalTask!!.run(conf)
                }
                additionalTask!!.run()
            }
        return applicableAppliedChanges.map { it.model.id }
    }

    override fun afterExecution(executionStep: ExecutionStep<OfferRetargeting>) {
        additionalTask!!.run()
    }

    override val modificationModelProperty: ModelProperty<OfferRetargetingModification, List<ModelChanges<OfferRetargeting>>>
        get() = OFFER_RETARGETING_UPDATE

    override val modifyingModels: List<ModelChanges<OfferRetargeting>> =
        offerRetargetingModelChangesList

    override val modificationResultModelProperty: ModelProperty<OfferRetargetingModificationResult, List<Long>>
        get() = OFFER_RETARGETING_UPDATE_RESULT

    override val resultOfModification: List<Long>
        get() = result.get().result.map { it.result }
}
