package ru.yandex.direct.core.entity.offerretargeting.validation

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceContextIsAcceptedForStrategy
import ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceSearchIsAcceptedForStrategy
import ru.yandex.direct.core.entity.bids.validation.BidsConstraints.strategyIsSet
import ru.yandex.direct.core.entity.bids.validation.PriceValidator
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainerWithExistentAdGroups
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingAddContainerWithNonExistentAdGroups
import ru.yandex.direct.core.entity.offerretargeting.container.OfferRetargetingUpdateContainer
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting.AD_GROUP_ID
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting.ID
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting.IS_SUSPENDED
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting.PRICE
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting.PRICE_CONTEXT
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingDefects.offerRetargetingAlreadyDeleted
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingDefects.offerRetargetingAlreadySuspended
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingDefects.offerRetargetingCantBeUsedInAutoBudgetCampaign
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingDefects.offerRetargetingNotSuspended
import ru.yandex.direct.core.entity.offerretargeting.validation.OfferRetargetingDefects.tooManyOfferRetargetingsInAdGroup
import ru.yandex.direct.core.validation.defects.RightsDefects
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.constraint.CommonConstraints.inSet
import ru.yandex.direct.validation.constraint.CommonConstraints.notInSet
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.ModelChangesValidationTool
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateModel
import ru.yandex.direct.validation.util.validateModelChanges
import java.math.BigDecimal

@Service
class OfferRetargetingValidationService @Autowired constructor(
    private val campaignSubObjectAccessCheckerFactory: CampaignSubObjectAccessCheckerFactory
) {
    private val modelChangesPreValidationTool: ModelChangesValidationTool = ModelChangesValidationTool.builder()
        .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION).build()

    /**
     * Вадидация операции добавления офферных ретаргетингов в существующие группы
     */
    fun validateAddOfferRetargetings(
        preValidationResult: ValidationResult<List<OfferRetargeting>, Defect<*>>,
        container: OfferRetargetingAddContainerWithExistentAdGroups,
    ): ValidationResult<List<OfferRetargeting>, Defect<*>> {
        return validateList(preValidationResult) {
            val offerRetargetingCountByAdGroupIds = preValidationResult.value
                .groupingBy { it.adGroupId }
                .eachCount()

            val adGroupIds: Set<Long> = offerRetargetingCountByAdGroupIds.keys

            val checker = campaignSubObjectAccessCheckerFactory.newAdGroupChecker(
                container.operatorUid,
                container.clientId,
                adGroupIds
            )
            val adGroupAccessValidator = checker.createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)

            checkEachBy { offerRetargeting -> validateAdGroupId(offerRetargeting, container, adGroupAccessValidator) }

            checkEachBy(
                { offerRetargeting ->
                    val adGroupId = offerRetargeting.adGroupId
                    val campaign = container.getCampaignByAdGroupId(adGroupId)!!
                    val offerRetargetingsCount = offerRetargetingCountByAdGroupIds.getValue(adGroupId)
                    validateAddOfferRetargeting(offerRetargeting, campaign, offerRetargetingsCount)
                },
                When.isValid()
            )
        }
    }

    /**
     * Валидация добавления офферных ретаргетингов в группы, которые еще не созданы
     */
    fun validateAddOfferRetargetings(
        preValidationResult: ValidationResult<List<OfferRetargeting>, Defect<*>>,
        container: OfferRetargetingAddContainerWithNonExistentAdGroups
    ) = validateList(preValidationResult) {
        val newOfferRetargetingCountByAdGroupIndex: Map<Int, Int> = container.adGroupInfos.values
            .groupingBy { it.adGroupIndex }
            .eachCount()

        checkEachBy { index: Int, offerRetargeting: OfferRetargeting ->
            val campaign: Campaign = container.getCampaignByOfferRetargetingIndex(index)!!
            val adGroupIndex: Int = container.adGroupInfos[index]!!.adGroupIndex
            val offerRetargetingsCount = newOfferRetargetingCountByAdGroupIndex[adGroupIndex]!!
            validateAddOfferRetargeting(offerRetargeting, campaign, offerRetargetingsCount)
        }
    }

    private fun validateAddOfferRetargeting(
        offerRetargeting: OfferRetargeting,
        campaign: Campaign,
        offerRetargetingsCount: Int,
    ) = validateModel(offerRetargeting) {
        checkBy({ validateStrategy(it, campaign) }, When.isValid())
        check(offerRetargetingsNoMoreThanMax(offerRetargetingsCount), When.isValid())
        check(campaignTypeIsSupported(campaign), When.isValid())
    }

    private fun validateAdGroupId(
        offerRetargeting: OfferRetargeting,
        container: OfferRetargetingAddContainerWithExistentAdGroups,
        adGroupAccessValidator: CampaignSubObjectAccessValidator
    ) = validateModel(offerRetargeting) {
        item(AD_GROUP_ID)
            .check(notNull())
            .checkBy(adGroupAccessValidator, When.isValid())
            .check(
                inSet(container.campaignIdsByAdGroupIds.keys),
                objectNotFound(), When.isValid()
            )
    }


    private fun campaignTypeIsSupported(campaign: Campaign): Constraint<OfferRetargeting, Defect<*>> =
        fromPredicate(
            {
                val type = campaign.type
                type == CampaignType.TEXT || type == CampaignType.MOBILE_CONTENT || type == CampaignType.CONTENT_PROMOTION
            },
            campaignTypeNotSupported()
        )


    /**
     * Проверяем, что ставки выставлены правильно в соответствии со стратегией кампании
     */
    private fun validateStrategy(offerRetargeting: OfferRetargeting, campaign: Campaign) =
        validateModel(offerRetargeting) {
            val strategy = campaign.strategy
            check(strategyIsSet(strategy))
            val currency = campaign.currency.currency
            val whenStrategyIsSet = When.isTrue<BigDecimal, Defect<*>>(strategy != null)
            item(PRICE)
                .checkBy(PriceValidator(currency, AdGroupType.BASE), When.notNull())
                .weakCheck(priceSearchIsAcceptedForStrategy(strategy), whenStrategyIsSet)
            item(PRICE_CONTEXT)
                .checkBy(PriceValidator(currency, AdGroupType.BASE), When.notNull())
                .weakCheck(priceContextIsAcceptedForStrategy(strategy), whenStrategyIsSet)
        }


    private fun offerRetargetingsNoMoreThanMax(count: Int): Constraint<OfferRetargeting, Defect<*>> =
        fromPredicate({ count <= MAX_OFFER_RETARGETINGS_IN_GROUP }, tooManyOfferRetargetingsInAdGroup())

    fun validateDeleteOfferRetargetings(
        offerRetargetingIds: List<Long>,
        offerRetargetingsById: Map<Long, OfferRetargeting>,
        operatorUid: Long,
        clientId: ClientId
    ) = validateList(offerRetargetingIds) {
        checkEach(unique())

        val existClientsOfferRetargetingIds = offerRetargetingsById.keys

        checkEach(inSet(existClientsOfferRetargetingIds), objectNotFound())
        checkEach(offerRetargetingIsNotDeleted(offerRetargetingsById), When.isValid())

        val accessConstraint: Constraint<Long, Defect<*>> = campaignSubObjectAccessCheckerFactory
            .newOfferRetargetingChecker(operatorUid, clientId, offerRetargetingIds)
            .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
            .accessConstraint

        checkEach(accessConstraint, When.isValid())
    }

    private fun offerRetargetingIsNotDeleted(
        clientsOfferRetargetings: Map<Long, OfferRetargeting>
    ): Constraint<Long, Defect<*>> = fromPredicate(
        { id -> !clientsOfferRetargetings.getValue(id).isDeleted },
        offerRetargetingAlreadyDeleted()
    )

    fun validateUpdateOfferRetargetings(
        preValidationResult: ValidationResult<List<OfferRetargeting>, Defect<*>>,
        container: OfferRetargetingUpdateContainer,
    ) = validateList(preValidationResult) {
        val adGroupIds = preValidationResult.value.mapNotNull { it.adGroupId }
        val checker = campaignSubObjectAccessCheckerFactory.newAdGroupChecker(
            container.operatorUid,
            container.clientId, adGroupIds
        )
        val adGroupAccessValidator = checker.createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)

        checkEachBy(
            { offerRet -> validateUpdateOfferRetargeting(offerRet, container, adGroupAccessValidator) },
            When.isValid()
        )
    }

    private fun validateUpdateOfferRetargeting(
        offerRetargeting: OfferRetargeting,
        container: OfferRetargetingUpdateContainer,
        adGroupAccessValidator: CampaignSubObjectAccessValidator,
    ) = validateModel(offerRetargeting) {
        item(AD_GROUP_ID)
            .check(notNull())
            .checkBy(adGroupAccessValidator, When.isValid())

        checkBy(
            { offerRet -> validateStrategy(offerRet, container.getCampaignByOfferRetargetingId(offerRet.id)) },
            When.isValid()
        )
    }

    fun preValidateUpdateOfferRetargetings(
        modelChanges: List<ModelChanges<OfferRetargeting>>,
        container: OfferRetargetingUpdateContainer
    ): ValidationResult<List<ModelChanges<OfferRetargeting>>, Defect<*>> {
        val validationResult = modelChangesPreValidationTool.validateModelChangesList(
            modelChanges, container.offerRetargetingsByIds.keys
        )
        val existingOfferRetargetings = container.offerRetargetingsByIds
        val suspendedIds = existingOfferRetargetings.values.asSequence()
            .filter { obj: OfferRetargeting -> obj.isSuspended }
            .map { obj: OfferRetargeting -> obj.id }
            .toSet()

        return validateList(validationResult) {
            checkEachBy(
                { changes -> preValidateUpdateOfferRetargeting(container, changes, suspendedIds) },
                When.isValid()
            )
        }
    }

    private fun preValidateUpdateOfferRetargeting(
        container: OfferRetargetingUpdateContainer,
        modelChanges: ModelChanges<OfferRetargeting>,
        suspendedIds: Set<Long>
    ): ValidationResult<ModelChanges<OfferRetargeting>, Defect<*>> = validateModelChanges(modelChanges) {
        val campaign = container.getCampaignByOfferRetargetingId(modelChanges.id)
        val isAutobudget = campaign.autobudget
        check(
            offerRetargetingPriceIsNotChanged(offerRetargetingCantBeUsedInAutoBudgetCampaign()),
            When.isTrue(isAutobudget)
        )
        val suspendChange = modelChanges.getPropIfChanged(IS_SUSPENDED)
        if (suspendChange != null) {
            // Проверка на изменение флага c тем же значением
            item(modelChanges.id, ID.name()).apply {
                if (suspendChange) {
                    weakCheck(notInSet(suspendedIds), offerRetargetingAlreadySuspended())
                } else {
                    weakCheck(inSet(suspendedIds), offerRetargetingNotSuspended())
                }
            }
        }
    }

    private fun offerRetargetingPriceIsNotChanged(
        defect: Defect<Unit>,
    ): Constraint<ModelChanges<OfferRetargeting>, Defect<*>> {
        val priceIsNotChanged = { changes: ModelChanges<OfferRetargeting> ->
            val searchPriceIsNotChanged = (!changes.isPropChanged(PRICE)
                || changes.getChangedProp(PRICE) == null)
            val contextPriceIsNotChanged = (!changes.isPropChanged(PRICE_CONTEXT)
                || changes.getChangedProp(PRICE_CONTEXT) == null)
            searchPriceIsNotChanged && contextPriceIsNotChanged
        }
        return fromPredicate(priceIsNotChanged, defect)
    }
}

const val MAX_OFFER_RETARGETINGS_IN_GROUP = 1

private const val MAX_ELEMENTS_PER_OPERATION = 10000

private val ACCESS_DEFECTS: CampaignAccessDefects = AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
    .withTypeNotAllowable(objectNotFound())
    .withNotVisible(objectNotFound())
    .withTypeNotSupported(campaignTypeNotSupported())
    .withNoRights(RightsDefects.noRights())
    .build()
