package ru.yandex.direct.core.entity.offlinereport.repository;

import java.nio.charset.StandardCharsets;
import java.util.List;

import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.offlinereport.model.OfflineReport;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportState;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportType;
import ru.yandex.direct.dbschema.ppc.enums.OfflineReportsReportState;
import ru.yandex.direct.dbschema.ppc.enums.OfflineReportsReportType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.tables.OfflineReports.OFFLINE_REPORTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class OfflineReportRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<OfflineReport> jooqMapper;

    public OfflineReportRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(OfflineReport::new)
                .map(property(OfflineReport.REPORT_ID, OFFLINE_REPORTS.OFFLINE_REPORT_ID))
                .map(property(OfflineReport.UID, OFFLINE_REPORTS.UID))
                .map(property(OfflineReport.SCHEDULED_AT, OFFLINE_REPORTS.SCHEDULED_AT))
                .map(convertibleProperty(OfflineReport.REPORT_STATE, OFFLINE_REPORTS.REPORT_STATE,
                        OfflineReportState::fromSource, OfflineReportState::toSource))
                .map(convertibleProperty(OfflineReport.REPORT_TYPE, OFFLINE_REPORTS.REPORT_TYPE,
                        OfflineReportType::fromSource, OfflineReportType::toSource))
                .map(convertibleProperty(OfflineReport.ARGS, OFFLINE_REPORTS.ARGS,
                        blob -> blob == null ? "" : new String(blob, StandardCharsets.UTF_8),
                        str -> str.getBytes(StandardCharsets.UTF_8)))
                .map(property(OfflineReport.REPORT_URL, OFFLINE_REPORTS.REPORT_URL))
                .build();
    }

    public Integer getPendingReportsCount(int shard, Long uid) {
        return dslContextProvider.ppc(shard)
                .selectCount()
                .from(OFFLINE_REPORTS)
                .where(OFFLINE_REPORTS.UID.eq(uid).and(OFFLINE_REPORTS.REPORT_STATE.eq(OfflineReportsReportState.New)))
                .fetchOne().value1();
    }

    public List<OfflineReport> getOfflineReports(int shard, Long uid, OfflineReportsReportType type) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(OFFLINE_REPORTS)
                .where(OFFLINE_REPORTS.UID.eq(uid)
                        .and(OFFLINE_REPORTS.REPORT_TYPE.eq(type)))
                .orderBy(OFFLINE_REPORTS.OFFLINE_REPORT_ID.desc())
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Добавить отчет в список
     *
     * @param shard         шард
     * @param offlineReport параметры отчета
     */
    public void addOfflineReport(int shard, OfflineReport offlineReport) {
        new InsertHelper<>(dslContextProvider.ppc(shard), OFFLINE_REPORTS)
                .add(jooqMapper, offlineReport)
                .execute();
    }

    /**
     * Получить параметры отчета по его id
     *
     * @param shard    шард
     * @param reportId идентификатор отчета
     * @return параметры отчета или {@code null}
     */
    public OfflineReport getOfflineReport(int shard, long reportId) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(OFFLINE_REPORTS)
                .where(OFFLINE_REPORTS.OFFLINE_REPORT_ID.eq(reportId))
                .fetchOne(jooqMapper::fromDb);
    }

    public void markReport(int shard, Long reportId, OfflineReportsReportState state) {
        dslContextProvider.ppc(shard)
                .update(OFFLINE_REPORTS)
                .set(OFFLINE_REPORTS.REPORT_STATE, state)
                .where(OFFLINE_REPORTS.OFFLINE_REPORT_ID.eq(reportId))
                .execute();
    }

    public void markReportReady(int shard, Long reportId, String reportUrl) {
        dslContextProvider.ppc(shard)
                .update(OFFLINE_REPORTS)
                .set(OFFLINE_REPORTS.REPORT_STATE, OfflineReportsReportState.Ready)
                .set(OFFLINE_REPORTS.REPORT_URL, reportUrl)
                .where(OFFLINE_REPORTS.OFFLINE_REPORT_ID.eq(reportId))
                .execute();
    }
}
