package ru.yandex.direct.core.entity.offlinereport.service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReport;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportJobParams;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportState;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportType;
import ru.yandex.direct.core.entity.offlinereport.repository.OfflineReportRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbschema.ppc.enums.OfflineReportsReportState;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.OFFLINE_REPORT;

@Service
public class OfflineReportService {
    private static final Logger logger = LoggerFactory.getLogger(OfflineReportService.class);
    private static final int MAX_PENDING_REPORT_COUNT = 20;
    private static final Map<OfflineReportType, FeatureName> REQUIRED_FEATURES_BY_REPORT_TYPE = Map.of(
            OfflineReportType.AGENCY_KPI, FeatureName.AGENCY_KPI_OFFLINE_REPORT_ENABLED
    );

    private final OfflineReportRepository offlineReportRepository;
    private final DbQueueRepository dbQueueRepository;
    private final ShardHelper shardHelper;
    private final ClientService clientService;
    private final FeatureService featureService;

    public OfflineReportService(OfflineReportRepository offlineReportRepository,
                                DbQueueRepository dbQueueRepository, ShardHelper shardHelper,
                                ClientService clientService,
                                FeatureService featureService) {
        this.offlineReportRepository = offlineReportRepository;
        this.dbQueueRepository = dbQueueRepository;
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.featureService = featureService;
    }

    /**
     * список оффлайн отчётов определенного типа.
     *
     * @param operator   оператор
     * @param reportType тип отчета
     * @return список отчетов
     */
    public List<OfflineReport> getAgencyOfflineReports(User operator, OfflineReportType reportType) {
        var clientId = operator.getClientId();
        checkReportTypeAccessibility(clientId, reportType);

        int shard = clientService.getShardByClientIdStrictly(clientId);
        //подменить URL если useAccelRedirect
        var offlineReportsReportType = OfflineReportType.toSource(reportType);
        return offlineReportRepository.getOfflineReports(shard, operator.getUid(), offlineReportsReportType)
                .stream()
                .map(report -> report.withReportUrl(externalReportUrl(report, operator)))
                .collect(Collectors.toList());
    }

    private String externalReportUrl(OfflineReport report, User operator) {
        if (report.getReportUrl() == null) {
            return null;
        }
        return String.format("/web-api/offline_report/download?report_id=%s&ulogin=%s",
                report.getReportId(), operator.getLogin());
    }

    /**
     * запросить создание нового оффлайн отчёта с периодом с точностью до месяца
     *
     * @param operator   оператор
     * @param reportType тип отчёта
     * @param monthFrom  Начало периода. Месяц в формате yyyymm
     * @param monthTo    Окончание периода. Месяц в формате yyyymm
     * @return новый отчёт с id
     */
    public OfflineReport createOfflineReportWithMonthlyInterval(User operator, User agencyUser,
                                                                OfflineReportType reportType,
                                                                String monthFrom, String monthTo) {
        return createOfflineReport(operator, agencyUser, reportType, false, monthFrom, monthTo, null, null);
    }

    /**
     * запросить создание нового оффлайн отчёта с периодом с точностью до дня
     *
     * @param operator   оператор
     * @param reportType тип отчёта
     * @param dateFrom   Начало периода
     * @param dateTo     Окончание периода
     * @return новый отчёт с id
     */
    public OfflineReport createOfflineReportWithDailyInterval(User operator, User agencyUser,
                                                              OfflineReportType reportType,
                                                              LocalDate dateFrom, LocalDate dateTo) {
        return createOfflineReport(operator, agencyUser, reportType, true, null, null, dateFrom, dateTo);
    }

    private OfflineReport createOfflineReport(User operator, User agencyUser,
                                              OfflineReportType reportType,
                                              boolean isWithDailyInterval,
                                              String monthFrom, String monthTo,
                                              LocalDate dateFrom, LocalDate dateTo) {
        var clientId = operator.getClientId();
        checkReportTypeAccessibility(clientId, reportType);

        int shard = clientService.getShardByClientIdStrictly(clientId);
        //количество отчётов в очереди на одного клиента ограничить 20. Защита от флуда.
        if (offlineReportRepository.getPendingReportsCount(shard, operator.getUid()) > MAX_PENDING_REPORT_COUNT) {
            throw new IllegalStateException("reach max pending report count: " + MAX_PENDING_REPORT_COUNT);
        }

        Long reportId = shardHelper.generateOfflineReportIds(1).get(0);
        //задача в DbQueue
        Long agencyId = agencyUser == null ? clientId.asLong() : agencyUser.getClientId().asLong();
        OfflineReportJobParams jobParams = isWithDailyInterval
                ? new OfflineReportJobParams(reportId, reportType, agencyId, dateFrom, dateTo)
                : new OfflineReportJobParams(reportId, reportType, agencyId, monthFrom, monthTo);
        Long jobId = dbQueueRepository
                .insertJob(shard, OFFLINE_REPORT, clientId, operator.getUid(), jobParams).getId();
        logger.info("offlineReportCreate jobId={}", jobId);

        //положить в таблицу offline_reports
        OfflineReport report = new OfflineReport()
                .withReportId(reportId)
                .withReportState(OfflineReportState.NEW)
                .withScheduledAt(LocalDateTime.now())
                .withUid(operator.getUid())
                .withReportType(reportType)
                .withArgs(getArgsJSON(isWithDailyInterval, monthFrom, monthTo, dateFrom, dateTo));
        offlineReportRepository.addOfflineReport(shard, report);
        return offlineReportRepository.getOfflineReport(shard, reportId);
    }

    private String getArgsJSON(boolean isWithDailyInterval,
                               String monthFrom, String monthTo,
                               LocalDate dateFrom, LocalDate dateTo) {
        Map<String, String> args = isWithDailyInterval
                ? ImmutableMap.<String, String>builder()
                .put("dateFrom", dateFrom.toString())
                .put("dateTo", dateTo.toString())
                .build()
                : ImmutableMap.<String, String>builder()
                .put("monthFrom", monthFrom)
                .put("monthTo", monthTo)
                .build();
        try {
            return new ObjectMapper().writeValueAsString(args);
        } catch (JsonProcessingException e) {
            logger.error(e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }

    /**
     * Получить параметры отчета по его id
     *
     * @param shard    шард
     * @param reportId идентификатор отчета
     * @return параметры отчета или {@code null}
     */
    public OfflineReport getOfflineReport(int shard, long reportId) {
        return offlineReportRepository.getOfflineReport(shard, reportId);
    }

    public void markReportProcessing(int shard, Long reportId) {
        offlineReportRepository.markReport(shard, reportId, OfflineReportsReportState.Processing);
    }

    public void markReportFailed(int shard, Long reportId) {
        offlineReportRepository.markReport(shard, reportId, OfflineReportsReportState.Error);
    }

    public void markReportReady(int shard, Long reportId, String reportUrl) {
        offlineReportRepository.markReportReady(shard, reportId, reportUrl);
    }

    public OfflineReport getReport(Long reportId, ClientId clientId) {
        int shard = clientService.getShardByClientIdStrictly(clientId);
        var report = offlineReportRepository.getOfflineReport(shard, reportId);
        checkReportTypeAccessibility(clientId, report.getReportType());
        return report;
    }

    private void checkReportTypeAccessibility(ClientId clientId, OfflineReportType reportType) {
        var requiredFeature = REQUIRED_FEATURES_BY_REPORT_TYPE.getOrDefault(reportType, null);
        if (requiredFeature != null && !featureService.isEnabledForClientId(clientId, requiredFeature)) {
            var errorMessage = "Reports of type " + reportType + " are not accessible to the client";
            logger.warn(errorMessage + " " + clientId);
            throw new IllegalStateException(errorMessage);
        }
    }
}
